#include "opengl_21_tutorials_win32_framework.h"

// ----------------------------------------------------------------------------------------------------------------------------

CBuffer::CBuffer()
{
	SetDefaults();
}

CBuffer::~CBuffer()
{
	Empty();
}

void CBuffer::AddData(void *Data, int DataSize)
{
	int Remaining = BufferSize - Position;

	if(DataSize > Remaining)
	{
		BYTE *OldBuffer = Buffer;
		int OldBufferSize = BufferSize;

		int Needed = DataSize - Remaining;

		BufferSize += Needed > BUFFER_SIZE_INCREMENT ? Needed : BUFFER_SIZE_INCREMENT;

		Buffer = new BYTE[BufferSize];

		memcpy(Buffer, OldBuffer, OldBufferSize);

		delete [] OldBuffer;
	}

	memcpy(Buffer + Position, Data, DataSize);

	Position += DataSize;
}

void CBuffer::Empty()
{
	delete [] Buffer;

	SetDefaults();
}

void *CBuffer::GetData()
{
	return Buffer;
}

int CBuffer::GetDataSize()
{
	return Position;
}

void CBuffer::SetDefaults()
{
	Buffer = NULL;

	BufferSize = 0;
	Position = 0;
}

// ----------------------------------------------------------------------------------------------------------------------------

int gl_max_texture_size = 0, gl_max_texture_max_anisotropy_ext = 0;

// ----------------------------------------------------------------------------------------------------------------------------

CTexture::CTexture()
{
	Texture = 0;
}

CTexture::~CTexture()
{
}

CTexture::operator GLuint ()
{
	return Texture;
}

bool CTexture::LoadTexture2D(char *FileName)
{
	CString DirectoryFileName = ModuleDirectory + FileName;

	int Width, Height, BPP;

	FIBITMAP *dib = GetBitmap(DirectoryFileName, Width, Height, BPP);

	if(dib == NULL)
	{
		ErrorLog.Append("Error loading texture " + DirectoryFileName + "!\r\n");
		return false;
	}

	GLenum Format = 0;

	if(BPP == 32) Format = GL_BGRA;
	if(BPP == 24) Format = GL_BGR;

	if(Format == 0)
	{
		ErrorLog.Append("Unsupported texture format (%s)!\r\n", FileName);
		FreeImage_Unload(dib);
		return false;
	}

	Destroy();

	glGenTextures(1, &Texture);

	glBindTexture(GL_TEXTURE_2D, Texture);

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	if(GLEW_EXT_texture_filter_anisotropic)
	{
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY_EXT, gl_max_texture_max_anisotropy_ext);
	}

	glTexParameteri(GL_TEXTURE_2D, GL_GENERATE_MIPMAP, GL_TRUE);

	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, Width, Height, 0, Format, GL_UNSIGNED_BYTE, FreeImage_GetBits(dib));

	glBindTexture(GL_TEXTURE_2D, 0);

	FreeImage_Unload(dib);

	return true;
}

bool CTexture::LoadTextureCubeMap(char **FileNames)
{
	int Width, Height, BPP;

	FIBITMAP *dib[6];

	bool Error = false;
	
	for(int i = 0; i < 6; i++)
	{
		CString DirectoryFileName = ModuleDirectory + FileNames[i];

		dib[i] = GetBitmap(DirectoryFileName, Width, Height, BPP);

		if(dib[i] == NULL)
		{
			ErrorLog.Append("Error loading texture " + DirectoryFileName + "!\r\n");
			Error = true;
		}
	}

	if(Error)
	{
		for(int i = 0; i < 6; i++)
		{
			FreeImage_Unload(dib[i]);
		}

		return false;
	}

	GLenum Format = 0;
	
	if(BPP == 32) Format = GL_BGRA;
	if(BPP == 24) Format = GL_BGR;

	if(Format == 0)
	{
		ErrorLog.Append("Unsupported texture format (%s)!\r\n", FileNames[5]);

		for(int i = 0; i < 6; i++)
		{
			FreeImage_Unload(dib[i]);
		}

		return false;
	}

	Destroy();

	glGenTextures(1, &Texture);

	glBindTexture(GL_TEXTURE_CUBE_MAP, Texture);

	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	if(GLEW_EXT_texture_filter_anisotropic)
	{
		glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MAX_ANISOTROPY_EXT, gl_max_texture_max_anisotropy_ext);
	}

	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_GENERATE_MIPMAP, GL_TRUE);

	for(int i = 0; i < 6; i++)
	{
		glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_X + i, 0, GL_RGBA8, Width, Height, 0, Format, GL_UNSIGNED_BYTE, FreeImage_GetBits(dib[i]));
	}

	glBindTexture(GL_TEXTURE_CUBE_MAP, 0);

	for(int i = 0; i < 6; i++)
	{
		FreeImage_Unload(dib[i]);
	}

	return true;
}

void CTexture::Destroy()
{
	glDeleteTextures(1, &Texture);
	Texture = 0;
}

FIBITMAP *CTexture::GetBitmap(char *FileName, int &Width, int &Height, int &BPP)
{
	FREE_IMAGE_FORMAT fif = FreeImage_GetFileType(FileName);

	if(fif == FIF_UNKNOWN)
	{
		fif = FreeImage_GetFIFFromFilename(FileName);
	}

	if(fif == FIF_UNKNOWN)
	{
		return NULL;
	}

	FIBITMAP *dib = NULL;

	if(FreeImage_FIFSupportsReading(fif))
	{
		dib = FreeImage_Load(fif, FileName);
	}

	if(dib != NULL)
	{
		int OriginalWidth = FreeImage_GetWidth(dib);
		int OriginalHeight = FreeImage_GetHeight(dib);

		Width = OriginalWidth;
		Height = OriginalHeight;

		if(Width == 0 || Height == 0)
		{
			FreeImage_Unload(dib);
			return NULL;
		}

		BPP = FreeImage_GetBPP(dib);

		if(Width > gl_max_texture_size) Width = gl_max_texture_size;
		if(Height > gl_max_texture_size) Height = gl_max_texture_size;

		if(!GLEW_ARB_texture_non_power_of_two)
		{
			Width = 1 << (int)floor((log((float)Width) / log(2.0f)) + 0.5f); 
			Height = 1 << (int)floor((log((float)Height) / log(2.0f)) + 0.5f);
		}

		if(Width != OriginalWidth || Height != OriginalHeight)
		{
			FIBITMAP *rdib = FreeImage_Rescale(dib, Width, Height, FILTER_BICUBIC);
			FreeImage_Unload(dib);
			dib = rdib;
		}
	}

	return dib;
}

// ----------------------------------------------------------------------------------------------------------------------------

CShaderProgram::CShaderProgram()
{
	SetDefaults();
}

CShaderProgram::~CShaderProgram()
{
}

CShaderProgram::operator GLuint ()
{
	return Program;
}

bool CShaderProgram::Load(char *VertexShaderFileName, char *FragmentShaderFileName)
{
	bool Error = false;

	Destroy();

	Error |= ((VertexShader = LoadShader(VertexShaderFileName, GL_VERTEX_SHADER)) == 0);
	Error |= ((FragmentShader = LoadShader(FragmentShaderFileName, GL_FRAGMENT_SHADER)) == 0);

	if(Error)
	{
		Destroy();
		return false;
	}

	Program = glCreateProgram();
	glAttachShader(Program, VertexShader);
	glAttachShader(Program, FragmentShader);
	glLinkProgram(Program);

	int LinkStatus;
	glGetProgramiv(Program, GL_LINK_STATUS, &LinkStatus);

	if(LinkStatus == GL_FALSE)
	{
		ErrorLog.Append("Error linking program (%s, %s)!\r\n", VertexShaderFileName, FragmentShaderFileName);

		int InfoLogLength = 0;
		glGetProgramiv(Program, GL_INFO_LOG_LENGTH, &InfoLogLength);
	
		if(InfoLogLength > 0)
		{
			char *InfoLog = new char[InfoLogLength];
			int CharsWritten  = 0;
			glGetProgramInfoLog(Program, InfoLogLength, &CharsWritten, InfoLog);
			ErrorLog.Append(InfoLog);
			delete [] InfoLog;
		}

		Destroy();

		return false;
	}

	return true;
}

void CShaderProgram::Destroy()
{
	glDetachShader(Program, VertexShader);
	glDetachShader(Program, FragmentShader);

	glDeleteShader(VertexShader);
	glDeleteShader(FragmentShader);

	glDeleteProgram(Program);

	delete [] UniformLocations;
	delete [] AttribLocations;

	SetDefaults();
}

GLuint CShaderProgram::LoadShader(char *FileName, GLenum Type)
{
	CString DirectoryFileName = ModuleDirectory + FileName;

	FILE *File;

	if(fopen_s(&File, DirectoryFileName, "rb") != 0)
	{
		ErrorLog.Append("Error loading file " + DirectoryFileName + "!\r\n");
		return 0;
	}

	fseek(File, 0, SEEK_END);
	long Size = ftell(File);
	fseek(File, 0, SEEK_SET);
	char *Source = new char[Size + 1];
	fread(Source, 1, Size, File);
	fclose(File);
	Source[Size] = 0;

	GLuint Shader = glCreateShader(Type);

	glShaderSource(Shader, 1, (const char**)&Source, NULL);
	delete [] Source;
	glCompileShader(Shader);

	int CompileStatus;
	glGetShaderiv(Shader, GL_COMPILE_STATUS, &CompileStatus);

	if(CompileStatus == GL_FALSE)
	{
		ErrorLog.Append("Error compiling shader %s!\r\n", FileName);

		int InfoLogLength = 0;
		glGetShaderiv(Shader, GL_INFO_LOG_LENGTH, &InfoLogLength);
	
		if(InfoLogLength > 0)
		{
			char *InfoLog = new char[InfoLogLength];
			int CharsWritten  = 0;
			glGetShaderInfoLog(Shader, InfoLogLength, &CharsWritten, InfoLog);
			ErrorLog.Append(InfoLog);
			delete [] InfoLog;
		}

		glDeleteShader(Shader);

		return 0;
	}

	return Shader;
}

void CShaderProgram::SetDefaults()
{
	VertexShader = 0;
	FragmentShader = 0;

	Program = 0;

	UniformLocations = NULL;
	AttribLocations = NULL;
}

// ----------------------------------------------------------------------------------------------------------------------------

CCamera::CCamera()
{
	ViewMatrix = NULL;
	ViewMatrixInverse = NULL;

	X = vec3(1.0f, 0.0f, 0.0f);
	Y = vec3(0.0f, 1.0f, 0.0f);
	Z = vec3(0.0f, 0.0f, 1.0f);

	Position = vec3(0.0f, 0.0f, 5.0f);
	Reference = vec3(0.0f, 0.0f, 0.0f);
}

CCamera::~CCamera()
{
}

void CCamera::Look(const vec3 &Position, const vec3 &Reference, bool RotateAroundReference)
{
	this->Position = Position;
	this->Reference = Reference;

	Z = normalize(Position - Reference);
	X = normalize(cross(vec3(0.0f, 1.0f, 0.0f), Z));
	Y = cross(Z, X);

	if(!RotateAroundReference)
	{
		this->Reference = this->Position;
		this->Position += Z * 0.05f;
	}

	CalculateViewMatrix();
}

void CCamera::Move(const vec3 &Movement)
{
	Position += Movement;
	Reference += Movement;

	CalculateViewMatrix();
}

vec3 CCamera::OnKeys(BYTE Keys, float FrameTime)
{
	float Speed = 5.0f;

	if(Keys & 0x40) Speed *= 2.0f;
	if(Keys & 0x80) Speed *= 0.5f;

	float Distance = Speed * FrameTime;

	vec3 Up(0.0f, 1.0f, 0.0f);
	vec3 Right = X;
	vec3 Forward = cross(Up, Right);

	Up *= Distance;
	Right *= Distance;
	Forward *= Distance;

	vec3 Movement;

	if(Keys & 0x01) Movement += Forward;
	if(Keys & 0x02) Movement -= Forward;
	if(Keys & 0x04) Movement -= Right;
	if(Keys & 0x08) Movement += Right;
	if(Keys & 0x10) Movement += Up;
	if(Keys & 0x20) Movement -= Up;

	return Movement;
}

void CCamera::OnMouseMove(int dx, int dy)
{
	float Sensitivity = 0.25f;

	Position -= Reference;

	if(dx != 0)
	{
		float DeltaX = (float)dx * Sensitivity;

		X = rotate(X, DeltaX, vec3(0.0f, 1.0f, 0.0f));
		Y = rotate(Y, DeltaX, vec3(0.0f, 1.0f, 0.0f));
		Z = rotate(Z, DeltaX, vec3(0.0f, 1.0f, 0.0f));
	}

	if(dy != 0)
	{
		float DeltaY = (float)dy * Sensitivity;

		Y = rotate(Y, DeltaY, X);
		Z = rotate(Z, DeltaY, X);

		if(Y.y < 0.0f)
		{
			Z = vec3(0.0f, Z.y > 0.0f ? 1.0f : -1.0f, 0.0f);
			Y = cross(Z, X);
		}
	}

	Position = Reference + Z * length(Position);

	CalculateViewMatrix();
}

void CCamera::OnMouseWheel(float zDelta)
{
	Position -= Reference;

	if(zDelta < 0 && length(Position) < 500.0f)
	{
		Position += Position * 0.1f;
	}

	if(zDelta > 0 && length(Position) > 0.05f)
	{
		Position -= Position * 0.1f;
	}

	Position += Reference;

	CalculateViewMatrix();
}

void CCamera::SetViewMatrixPointer(float *ViewMatrix, float *ViewMatrixInverse)
{
	this->ViewMatrix = (mat4x4*)ViewMatrix;
	this->ViewMatrixInverse = (mat4x4*)ViewMatrixInverse;

	CalculateViewMatrix();
}

void CCamera::CalculateViewMatrix()
{
	if(ViewMatrix != NULL)
	{
		*ViewMatrix = mat4x4(X.x, Y.x, Z.x, 0.0f, X.y, Y.y, Z.y, 0.0f, X.z, Y.z, Z.z, 0.0f, -dot(X, Position), -dot(Y, Position), -dot(Z, Position), 1.0f);

		if(ViewMatrixInverse != NULL)
		{
			*ViewMatrixInverse = inverse(*ViewMatrix);
		}
	}
}

// ----------------------------------------------------------------------------------------------------------------------------

CCamera Camera;

// ----------------------------------------------------------------------------------------------------------------------------

COpenGLRenderer::COpenGLRenderer()
{
	WHMID = 0;

	WireFrame = false;
	Pause = false;

	DropRadius = 4.0f / 128.0f;

	Camera.SetViewMatrixPointer(&ViewMatrix, &ViewMatrixInverse);
}

COpenGLRenderer::~COpenGLRenderer()
{
}

bool COpenGLRenderer::Init()
{
	// ------------------------------------------------------------------------------------------------------------------------

	bool Error = false;

	// ------------------------------------------------------------------------------------------------------------------------

	if(!GLEW_ARB_texture_non_power_of_two)
	{
		ErrorLog.Append("GL_ARB_texture_non_power_of_two not supported!\r\n");
		Error = true;
	}

	if(!GLEW_ARB_texture_float)
	{
		ErrorLog.Append("GL_ARB_texture_float not supported!\r\n");
		Error = true;
	}

	if(!GLEW_EXT_framebuffer_object)
	{
		ErrorLog.Append("GL_EXT_framebuffer_object not supported!\r\n");
		Error = true;
	}

	// ------------------------------------------------------------------------------------------------------------------------

	char *PoolSkyCubeMapFileNames[] = {"pool\\right.jpg", "pool\\left.jpg", "pool\\bottom.jpg", "pool\\top.jpg", "pool\\front.jpg", "pool\\back.jpg"};

	Error |= !PoolSkyCubeMap.LoadTextureCubeMap(PoolSkyCubeMapFileNames);

	// ------------------------------------------------------------------------------------------------------------------------

	Error |= !WaterAddDropProgram.Load("wateradddrop.vs", "wateradddrop.fs");
	Error |= !WaterHeightMapProgram.Load("waterheightmap.vs", "waterheightmap.fs");
	Error |= !WaterNormalMapProgram.Load("waternormalmap.vs", "waternormalmap.fs");
	Error |= !PhotonProgram.Load("photon.vs", "photon.fs");
	Error |= !CubeMapHBlurProgram.Load("cube_map_blur.vs", "cube_map_hblur.fs");
	Error |= !CubeMapVBlurProgram.Load("cube_map_blur.vs", "cube_map_vblur.fs");
	Error |= !PoolSkyProgram.Load("poolsky.vs", "poolsky.fs");
	Error |= !WaterProgram.Load("water.vs", "water.fs");

	// ------------------------------------------------------------------------------------------------------------------------

	if(Error)
	{
		return false;
	}

	// ------------------------------------------------------------------------------------------------------------------------

	vec3 LightPosition = vec3(0.0f, 1.0f, 0.0f);

	vec3 CubeMapNormals[6] = {
		vec3(-1.0f, 0.0f, 0.0f),
		vec3(1.0f, 0.0f, 0.0f),
		vec3(0.0f, -1.0f, 0.0f),
		vec3(0.0f, 1.0f, 0.0f),
		vec3(0.0f, 0.0f, -1.0f),
		vec3(0.0f, 0.0f, 1.0f),
	};

	mat4x4 BiasScaleMatrix = scale(1.0f / 2.0f, 1.0f / 3.0f, 1.0f) * BiasMatrix;

	mat4x4 PhotonsWorldToTextureMatrices[6] = {
		translate(0.0f / 2.0f, 0.0f / 3.0f, 1.0f) * BiasScaleMatrix * mat4x4(vec4(0.0f, 0.0f, -1.0f, 0.0f), vec4(0.0f, 1.0f, 0.0f, 0.0f), vec4(1.0f, 0.0f, 0.0f, 0.0f), vec4(0.0f, 0.0f, 0.0f, 1.0f)),
		translate(1.0f / 2.0f, 0.0f / 3.0f, 1.0f) * BiasScaleMatrix * mat4x4(vec4(0.0f, 0.0f, 1.0f, 0.0f), vec4(0.0f, 1.0f, 0.0f, 0.0f), vec4(-1.0f, 0.0f, 0.0f, 0.0f), vec4(0.0f, 0.0f, 0.0f, 1.0f)),
		translate(0.0f / 2.0f, 1.0f / 3.0f, 1.0f) * BiasScaleMatrix * mat4x4(vec4(1.0f, 0.0f, 0.0f, 0.0f), vec4(0.0f, 0.0f, -1.0f, 0.0f), vec4(0.0f, 1.0f, 0.0f, 0.0f), vec4(0.0f, 0.0f, 0.0f, 1.0f)),
		translate(1.0f / 2.0f, 1.0f / 3.0f, 1.0f) * BiasScaleMatrix * mat4x4(vec4(1.0f, 0.0f, 0.0f, 0.0f), vec4(0.0f, 0.0f, 1.0f, 0.0f), vec4(0.0f, -1.0f, 0.0f, 0.0f), vec4(0.0f, 0.0f, 0.0f, 1.0f)),
		translate(0.0f / 2.0f, 2.0f / 3.0f, 1.0f) * BiasScaleMatrix * mat4x4(vec4(-1.0f, 0.0f, 0.0f, 0.0f), vec4(0.0f, 1.0f, 0.0f, 0.0f), vec4(0.0f, 0.0f, -1.0f, 0.0f), vec4(0.0f, 0.0f, 0.0f, 1.0f)),
		translate(1.0f / 2.0f, 2.0f / 3.0f, 1.0f) * BiasScaleMatrix * mat4x4(vec4(1.0f, 0.0f, 0.0f, 0.0f), vec4(0.0f, 1.0f, 0.0f, 0.0f), vec4(0.0f, 0.0f, 1.0f, 0.0f), vec4(0.0f, 0.0f, 0.0f, 1.0f))
	};

	// ------------------------------------------------------------------------------------------------------------------------

	glUseProgram(WaterHeightMapProgram);
	glUniform1f(glGetUniformLocation(WaterHeightMapProgram, "ODWHMR"), 1.0f / (float)WHMR);
	glUseProgram(0);

	glUseProgram(WaterNormalMapProgram);
	glUniform1f(glGetUniformLocation(WaterNormalMapProgram, "ODWNMR"), 1.0f / (float)WNMR);
	glUniform1f(glGetUniformLocation(WaterNormalMapProgram, "WMSDWNMRM2"), 2.0f / (float)WNMR * 2.0f);
	glUseProgram(0);

	glUseProgram(PhotonProgram);
	glUniform1i(glGetUniformLocation(PhotonProgram, "WaterHeightMap"), 0);
	glUniform1i(glGetUniformLocation(PhotonProgram, "WaterNormalMap"), 1);
	glUniform1i(glGetUniformLocation(PhotonProgram, "PhotonsTexture"), 2);
	glUniform3fv(glGetUniformLocation(PhotonProgram, "LightPosition"), 1, &LightPosition);
	glUniform3fv(glGetUniformLocation(PhotonProgram, "CubeMapNormals"), 6, (float*)CubeMapNormals);
	glUniformMatrix4fv(glGetUniformLocation(PhotonProgram, "PhotonsWorldToTextureMatrices"), 6, GL_FALSE, (float*)PhotonsWorldToTextureMatrices);
	glUseProgram(0);

	glUseProgram(PoolSkyProgram);
	glUniform1i(glGetUniformLocation(PoolSkyProgram, "PoolSkyCubeMap"), 0);
	glUniform1i(glGetUniformLocation(PoolSkyProgram, "PhotonsCubeMap"), 1);
	glUseProgram(0);

	glUseProgram(WaterProgram);
	glUniform1i(glGetUniformLocation(WaterProgram, "WaterHeightMap"), 0);
	glUniform1i(glGetUniformLocation(WaterProgram, "WaterNormalMap"), 1);
	glUniform1i(glGetUniformLocation(WaterProgram, "PoolSkyCubeMap"), 2);
	glUniform1i(glGetUniformLocation(WaterProgram, "PhotonsCubeMap"), 3);
	glUniform1f(glGetUniformLocation(WaterProgram, "ODWMS"), 1.0f / 2.0f);
	glUniform3fv(glGetUniformLocation(WaterProgram, "LightPosition"), 1, &LightPosition);
	glUniform3fv(glGetUniformLocation(WaterProgram, "CubeMapNormals"), 6, (float*)CubeMapNormals);
	glUseProgram(0);

	// ------------------------------------------------------------------------------------------------------------------------

	WaterAddDropProgram.UniformLocations = new GLuint[2];
	WaterAddDropProgram.UniformLocations[0] = glGetUniformLocation(WaterAddDropProgram, "DropRadius");
	WaterAddDropProgram.UniformLocations[1] = glGetUniformLocation(WaterAddDropProgram, "Position");

	CubeMapHBlurProgram.UniformLocations = new GLuint[1];
	CubeMapHBlurProgram.UniformLocations[0] = glGetUniformLocation(CubeMapHBlurProgram, "Offset");

	CubeMapVBlurProgram.UniformLocations = new GLuint[1];
	CubeMapVBlurProgram.UniformLocations[0] = glGetUniformLocation(CubeMapVBlurProgram, "Offset");

	WaterProgram.UniformLocations = new GLuint[1];
	WaterProgram.UniformLocations[0] = glGetUniformLocation(WaterProgram, "CameraPosition");

	// ------------------------------------------------------------------------------------------------------------------------

	glGenTextures(2, WaterHeightMaps);

	vec4 *Heights = new vec4[WHMR * WHMR];

	for(int i = 0; i < WHMR * WHMR; i++)
	{
		Heights[i] = vec4(0.0f, 0.0f, 0.0f, 0.0f);
	}

	for(int i = 0; i < 2; i++)
	{
		glBindTexture(GL_TEXTURE_2D, WaterHeightMaps[i]);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY_EXT, gl_max_texture_max_anisotropy_ext);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
		glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA16F, WHMR, WHMR, 0, GL_RGBA, GL_FLOAT, Heights);
		glGenerateMipmapEXT(GL_TEXTURE_2D);
		glBindTexture(GL_TEXTURE_2D, 0);
	}

	delete [] Heights;

	// ------------------------------------------------------------------------------------------------------------------------

	glGenTextures(1, &WaterNormalMap);

	vec4 *Normals = new vec4[WNMR * WNMR];

	for(int i = 0; i < WNMR * WNMR; i++)
	{
		Normals[i] = vec4(0.0f, 1.0f, 0.0f, 1.0f);
	}

	glBindTexture(GL_TEXTURE_2D, WaterNormalMap);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY_EXT, gl_max_texture_max_anisotropy_ext);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA16F, WNMR, WNMR, 0, GL_RGBA, GL_FLOAT, Normals);
	glGenerateMipmapEXT(GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D, 0);

	delete [] Normals;

	// ------------------------------------------------------------------------------------------------------------------------

	glGenTextures(1, &PhotonsTexture);

	vec4 *Colors = new vec4[PCMR * 2 * PCMR * 3];

	for(int i = 0; i < PCMR * 2 * PCMR * 3; i++)
	{
		Colors[i] = vec4(0.0f, 0.0f, 0.0f, 1.0f);
	}

	glBindTexture(GL_TEXTURE_2D, PhotonsTexture);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, PCMR * 2, PCMR * 3, 0, GL_RGBA, GL_FLOAT, Colors);
	glBindTexture(GL_TEXTURE_2D, 0);

	delete [] Colors;

	// ------------------------------------------------------------------------------------------------------------------------

	glGenTextures(2, PhotonsTempCubeMaps);

	for(int i = 0; i < 2; i++)
	{
		glBindTexture(GL_TEXTURE_CUBE_MAP, PhotonsTempCubeMaps[i]);
		glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
		glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

		for(int i = 0; i < 6; i++)
		{
			glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_X + i, 0, GL_RGBA8, PCMR, PCMR, 0, GL_RGBA, GL_UNSIGNED_BYTE, NULL);
		}

		glBindTexture(GL_TEXTURE_CUBE_MAP, 0);
	}

	// ------------------------------------------------------------------------------------------------------------------------

	glGenTextures(1, &PhotonsCubeMap);

	Colors = new vec4[PCMR * PCMR];

	for(int i = 0; i < PCMR * PCMR; i++)
	{
		Colors[i] = vec4(0.0f, 0.0f, 0.0f, 1.0f);
	}

	glBindTexture(GL_TEXTURE_CUBE_MAP, PhotonsCubeMap);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MAX_ANISOTROPY_EXT, gl_max_texture_max_anisotropy_ext);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

	for(int i = 0; i < 6; i++)
	{
		glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_X + i, 0, GL_RGBA8, PCMR, PCMR, 0, GL_RGBA, GL_FLOAT, Colors);
	}

	glGenerateMipmapEXT(GL_TEXTURE_CUBE_MAP);

	glBindTexture(GL_TEXTURE_CUBE_MAP, 0);

	delete [] Colors;

	// ------------------------------------------------------------------------------------------------------------------------

	glGenBuffers(1, &PhotonsVBO);

	int PMR = PCMR, PMRP1 = PMR + 1;

	PhotonsCount = PMRP1 * PMRP1;

	vec3 *Photons = new vec3[PhotonsCount];

	float WMSDPMR = 2.0f / (float)PMR;

	int i = 0;

	for(int y = 0; y <= PMR; y++)
	{
		for(int x = 0; x <= PMR; x++)
		{
			Photons[i++] = vec3((float)x * WMSDPMR - 1.0f, 0.0f, 1.0f - (float)y * WMSDPMR);
		}
	}

	glBindBuffer(GL_ARRAY_BUFFER, PhotonsVBO);
	glBufferData(GL_ARRAY_BUFFER, PhotonsCount * 12, Photons, GL_STATIC_DRAW);
	glBindBuffer(GL_ARRAY_BUFFER, 0);

	delete [] Photons;

	// ------------------------------------------------------------------------------------------------------------------------

	glGenBuffers(1, &PoolSkyVBO);

	float PoolSkyVertices[] =
	{	// x, y, z, x, y, z, x, y, z, x, y, z
		 1.0f, -1.0f, -1.0f,  1.0f, -1.0f,  1.0f,  1.0f,  1.0f,  1.0f,  1.0f,  1.0f, -1.0f, // +X
		-1.0f, -1.0f,  1.0f, -1.0f, -1.0f, -1.0f, -1.0f,  1.0f, -1.0f, -1.0f,  1.0f,  1.0f, // -X
		-1.0f,  1.0f, -1.0f,  1.0f,  1.0f, -1.0f,  1.0f,  1.0f,  1.0f, -1.0f,  1.0f,  1.0f, // +Y
		-1.0f, -1.0f,  1.0f,  1.0f, -1.0f,  1.0f,  1.0f, -1.0f, -1.0f, -1.0f, -1.0f, -1.0f, // -Y
		 1.0f, -1.0f,  1.0f, -1.0f, -1.0f,  1.0f, -1.0f,  1.0f,  1.0f,  1.0f,  1.0f,  1.0f, // +Z
		-1.0f, -1.0f, -1.0f,  1.0f, -1.0f, -1.0f,  1.0f,  1.0f, -1.0f, -1.0f,  1.0f, -1.0f  // -Z
	};

	glBindBuffer(GL_ARRAY_BUFFER, PoolSkyVBO);
	glBufferData(GL_ARRAY_BUFFER, 288, PoolSkyVertices, GL_STATIC_DRAW);
	glBindBuffer(GL_ARRAY_BUFFER, 0);

	// ------------------------------------------------------------------------------------------------------------------------

	glGenBuffers(1, &WaterVBO);

	int WMRP1 = WMR + 1;

	vec3 *Vertices = new vec3[WMRP1 * WMRP1];

	float WMSDWMR = 2.0f / (float)WMR;

	for(int y = 0; y <= WMR; y++)
	{
		for(int x = 0; x <= WMR; x++)
		{
			Vertices[WMRP1 * y + x].x = x * WMSDWMR - 1.0f;
			Vertices[WMRP1 * y + x].y = 0.0f;
			Vertices[WMRP1 * y + x].z = 1.0f - y * WMSDWMR;
		}
	}

	CBuffer Quads;

	for(int y = 0; y < WMR; y++)
	{
		int yp1 = y + 1;

		for(int x = 0; x < WMR; x++)
		{
			int xp1 = x + 1;

			int a = WMRP1 * y + x;
			int b = WMRP1 * y + xp1;
			int c = WMRP1 * yp1 + xp1;
			int d = WMRP1 * yp1 + x;

			Quads.AddData(&Vertices[a], 12);
			Quads.AddData(&Vertices[b], 12);
			Quads.AddData(&Vertices[c], 12);
			Quads.AddData(&Vertices[d], 12);
		}
	}

	glBindBuffer(GL_ARRAY_BUFFER, WaterVBO);
	glBufferData(GL_ARRAY_BUFFER, Quads.GetDataSize(), Quads.GetData(), GL_STATIC_DRAW);
	glBindBuffer(GL_ARRAY_BUFFER, 0);

	QuadsVerticesCount = Quads.GetDataSize() / 12;

	Quads.Empty();

	delete [] Vertices;

	// ------------------------------------------------------------------------------------------------------------------------

	glGenFramebuffersEXT(1, &FBO);

	// ------------------------------------------------------------------------------------------------------------------------

	Camera.Look(vec3(0.0f, 1.0f, 2.5f), vec3(0.0f, -0.5f, 0.0f), true);

	// ------------------------------------------------------------------------------------------------------------------------

	srand(GetTickCount());

	// ------------------------------------------------------------------------------------------------------------------------

	return true;
}

void COpenGLRenderer::Render(float FrameTime)
{
	// add drops --------------------------------------------------------------------------------------------------------------

	if(!Pause)
	{
		static DWORD LastTime = GetTickCount();

		DWORD Time = GetTickCount();

		if(Time - LastTime > 100)
		{
			LastTime = Time;

			AddDrop(2.0f * (float)rand() / (float)RAND_MAX - 1.0f, 1.0f - 2.0f * (float)rand() / (float)RAND_MAX, 4.0f / 128.0f * (float)rand() / (float)RAND_MAX);
		}
	}

	// update water surface and generate photon cube map ----------------------------------------------------------------------

	static DWORD LastTime = GetTickCount();

	DWORD Time = GetTickCount();

	if(Time - LastTime >= 16)
	{
		LastTime = Time;

		// update water height map --------------------------------------------------------------------------------------------

		glViewport(0, 0, WHMR, WHMR);

		GLuint whmid = (WHMID + 1) % 2;

		glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, FBO);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_2D, WaterHeightMaps[whmid], 0);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_TEXTURE_2D, 0, 0);

		glBindTexture(GL_TEXTURE_2D, WaterHeightMaps[WHMID]);
		glUseProgram(WaterHeightMapProgram);
		glBegin(GL_QUADS);
			glVertex2f(0.0f, 0.0f);
			glVertex2f(1.0f, 0.0f);
			glVertex2f(1.0f, 1.0f);
			glVertex2f(0.0f, 1.0f);
		glEnd();
		glUseProgram(0);
		glBindTexture(GL_TEXTURE_2D, 0);

		glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);

		glBindTexture(GL_TEXTURE_2D, WaterHeightMaps[whmid]);
		glGenerateMipmapEXT(GL_TEXTURE_2D);
		glBindTexture(GL_TEXTURE_2D, 0);

		++WHMID %= 2;

		// update water normal map --------------------------------------------------------------------------------------------

		glViewport(0, 0, WNMR, WNMR);

		glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, FBO);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_2D, WaterNormalMap, 0);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_TEXTURE_2D, 0, 0);

		glBindTexture(GL_TEXTURE_2D, WaterHeightMaps[WHMID]);
		glUseProgram(WaterNormalMapProgram);
		glBegin(GL_QUADS);
			glVertex2f(0.0f, 0.0f);
			glVertex2f(1.0f, 0.0f);
			glVertex2f(1.0f, 1.0f);
			glVertex2f(0.0f, 1.0f);
		glEnd();
		glUseProgram(0);
		glBindTexture(GL_TEXTURE_2D, 0);

		glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);

		glBindTexture(GL_TEXTURE_2D, WaterNormalMap);
		glGenerateMipmapEXT(GL_TEXTURE_2D);
		glBindTexture(GL_TEXTURE_2D, 0);

		// render photons into photons texture --------------------------------------------------------------------------------

		glViewport(0, 0, PCMR * 2, PCMR * 3);

		glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, FBO);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_2D, PhotonsTexture, 0);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_TEXTURE_2D, 0, 0);

		glClear(GL_COLOR_BUFFER_BIT);

		glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, WaterHeightMaps[WHMID]);
		glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_2D, WaterNormalMap);
		glActiveTexture(GL_TEXTURE2); glBindTexture(GL_TEXTURE_2D, PhotonsTexture);
		glUseProgram(PhotonProgram);
		glBindBuffer(GL_ARRAY_BUFFER, PhotonsVBO);
		glEnableClientState(GL_VERTEX_ARRAY);
		glVertexPointer(3, GL_FLOAT, 12, (void*)0);
		glDrawArrays(GL_POINTS, 0, PhotonsCount);
		glDisableClientState(GL_VERTEX_ARRAY);
		glBindBuffer(GL_ARRAY_BUFFER, 0);
		glUseProgram(0);
		glActiveTexture(GL_TEXTURE2); glBindTexture(GL_TEXTURE_2D, 0);
		glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_2D, 0);
		glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, 0);

		glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);

		// --------------------------------------------------------------------------------------------------------------------

		static vec2 PhotonsTextureCoords[] = {
			vec2(0.0f / 2.0f, 0.0f / 3.0f), vec2(1.0f / 2.0f, 0.0f / 3.0f), vec2(1.0f / 2.0f, 1.0f / 3.0f), vec2(0.0f / 2.0f, 1.0f / 3.0f),
			vec2(1.0f / 2.0f, 0.0f / 3.0f), vec2(2.0f / 2.0f, 0.0f / 3.0f), vec2(2.0f / 2.0f, 1.0f / 3.0f), vec2(1.0f / 2.0f, 1.0f / 3.0f),
			vec2(1.0f / 2.0f, 1.0f / 3.0f), vec2(2.0f / 2.0f, 1.0f / 3.0f), vec2(2.0f / 2.0f, 2.0f / 3.0f), vec2(1.0f / 2.0f, 2.0f / 3.0f),
			vec2(0.0f / 2.0f, 1.0f / 3.0f), vec2(1.0f / 2.0f, 1.0f / 3.0f), vec2(1.0f / 2.0f, 2.0f / 3.0f), vec2(0.0f / 2.0f, 2.0f / 3.0f),
			vec2(1.0f / 2.0f, 2.0f / 3.0f), vec2(2.0f / 2.0f, 2.0f / 3.0f), vec2(2.0f / 2.0f, 3.0f / 3.0f), vec2(1.0f / 2.0f, 3.0f / 3.0f),
			vec2(0.0f / 2.0f, 2.0f / 3.0f), vec2(1.0f / 2.0f, 2.0f / 3.0f), vec2(1.0f / 2.0f, 3.0f / 3.0f), vec2(0.0f / 2.0f, 3.0f / 3.0f),
		};

		// copy photons texture to photons temp cube map 1 --------------------------------------------------------------------

		glViewport(0, 0, PCMR, PCMR);

		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();

		glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();

		for(int i = 0; i < 6; i++)
		{
			glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, FBO);
			glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_CUBE_MAP_POSITIVE_X + i, PhotonsTempCubeMaps[0], 0);
			glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_TEXTURE_2D, 0, 0);

			glEnable(GL_TEXTURE_2D);
			glBindTexture(GL_TEXTURE_2D, PhotonsTexture);
			glBegin(GL_QUADS);
				glTexCoord2fv(&PhotonsTextureCoords[i * 4 + 0]); glVertex2f(-1.0f, -1.0f);
				glTexCoord2fv(&PhotonsTextureCoords[i * 4 + 1]); glVertex2f(1.0f, -1.0f);
				glTexCoord2fv(&PhotonsTextureCoords[i * 4 + 2]); glVertex2f(1.0f, 1.0f);
				glTexCoord2fv(&PhotonsTextureCoords[i * 4 + 3]); glVertex2f(-1.0f, 1.0f);
			glEnd();
			glBindTexture(GL_TEXTURE_2D, 0);
			glDisable(GL_TEXTURE_2D);

			glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
		}

		// --------------------------------------------------------------------------------------------------------------------

		static vec3 PhotonsCubeMapCoords[] = {
			vec3( 1.0f,  1.0f,  1.0f), vec3( 1.0f,  1.0f, -1.0f), vec3( 1.0f, -1.0f, -1.0f), vec3( 1.0f, -1.0f,  1.0f),
			vec3(-1.0f,  1.0f, -1.0f), vec3(-1.0f,  1.0f,  1.0f), vec3(-1.0f, -1.0f,  1.0f), vec3(-1.0f, -1.0f, -1.0f),
			vec3(-1.0f, -1.0f,  1.0f), vec3( 1.0f, -1.0f,  1.0f), vec3( 1.0f, -1.0f, -1.0f), vec3(-1.0f, -1.0f, -1.0f),
			vec3(-1.0f,  1.0f, -1.0f), vec3( 1.0f,  1.0f, -1.0f), vec3( 1.0f,  1.0f,  1.0f), vec3(-1.0f,  1.0f,  1.0f),
			vec3( 1.0f,  1.0f, -1.0f), vec3(-1.0f,  1.0f, -1.0f), vec3(-1.0f, -1.0f, -1.0f), vec3( 1.0f, -1.0f, -1.0f),
			vec3(-1.0f,  1.0f,  1.0f), vec3( 1.0f,  1.0f,  1.0f), vec3( 1.0f, -1.0f,  1.0f), vec3(-1.0f, -1.0f,  1.0f)
		};

		static float ODPCMR = 1.0f / (float)PCMR;

		static vec3 PhotonsCubeMapHorizontalOffsets[] = {
			vec3(0.0f, 0.0f, -ODPCMR), vec3(0.0f, 0.0f, ODPCMR),
			vec3(ODPCMR, 0.0f, 0.0f), vec3(ODPCMR, 0.0f, 0.0f),
			vec3(-ODPCMR, 0.0f, 0.0f), vec3(ODPCMR, 0.0f, 0.0f)
		};

		static vec3 PhotonsCubeMapVerticalOffsets[] = {
			vec3(0.0f, -ODPCMR, 0.0f), vec3(0.0f, -ODPCMR, 0.0f),
			vec3(0.0f, 0.0f, -ODPCMR), vec3(0.0f, 0.0f, ODPCMR),
			vec3(0.0f, -ODPCMR, 0.0f), vec3(0.0f, -ODPCMR, 0.0f)
		};

		// blur photons temp cube map 1 horizontally --------------------------------------------------------------------------

		for(int i = 0; i < 6; i++)
		{
			glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, FBO);
			glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_CUBE_MAP_POSITIVE_X + i, PhotonsTempCubeMaps[1], 0);
			glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_TEXTURE_2D, 0, 0);

			glBindTexture(GL_TEXTURE_CUBE_MAP, PhotonsTempCubeMaps[0]);
			glUseProgram(CubeMapHBlurProgram);
			glUniform3fv(CubeMapHBlurProgram.UniformLocations[0], 1, &PhotonsCubeMapHorizontalOffsets[i]);
			glBegin(GL_QUADS);
				glTexCoord3fv(&PhotonsCubeMapCoords[i * 4 + 0]); glVertex2f(-1.0f, -1.0f);
				glTexCoord3fv(&PhotonsCubeMapCoords[i * 4 + 1]); glVertex2f(1.0f, -1.0f);
				glTexCoord3fv(&PhotonsCubeMapCoords[i * 4 + 2]); glVertex2f(1.0f, 1.0f);
				glTexCoord3fv(&PhotonsCubeMapCoords[i * 4 + 3]); glVertex2f(-1.0f, 1.0f);
			glEnd();
			glUseProgram(0);
			glBindTexture(GL_TEXTURE_CUBE_MAP, 0);

			glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
		}

		// blur photons temp cube map 2 vertically ----------------------------------------------------------------------------

		for(int i = 0; i < 6; i++)
		{
			glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, FBO);
			glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_CUBE_MAP_POSITIVE_X + i, PhotonsCubeMap, 0);
			glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_TEXTURE_2D, 0, 0);

			glBindTexture(GL_TEXTURE_CUBE_MAP, PhotonsTempCubeMaps[1]);
			glUseProgram(CubeMapVBlurProgram);
			glUniform3fv(CubeMapVBlurProgram.UniformLocations[0], 1, &PhotonsCubeMapVerticalOffsets[i]);
			glBegin(GL_QUADS);
				glTexCoord3fv(&PhotonsCubeMapCoords[i * 4 + 0]); glVertex2f(-1.0f, -1.0f);
				glTexCoord3fv(&PhotonsCubeMapCoords[i * 4 + 1]); glVertex2f(1.0f, -1.0f);
				glTexCoord3fv(&PhotonsCubeMapCoords[i * 4 + 2]); glVertex2f(1.0f, 1.0f);
				glTexCoord3fv(&PhotonsCubeMapCoords[i * 4 + 3]); glVertex2f(-1.0f, 1.0f);
			glEnd();
			glUseProgram(0);
			glBindTexture(GL_TEXTURE_CUBE_MAP, 0);

			glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
		}

		// generate mipmaps ---------------------------------------------------------------------------------------------------

		glBindTexture(GL_TEXTURE_CUBE_MAP, PhotonsCubeMap);
		glGenerateMipmapEXT(GL_TEXTURE_CUBE_MAP);
		glBindTexture(GL_TEXTURE_CUBE_MAP, 0);
	}

	// render pool sky mesh ---------------------------------------------------------------------------------------------------

	glViewport(0, 0, Width, Height);

	glMatrixMode(GL_PROJECTION);
	glLoadMatrixf(&ProjectionMatrix);

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glEnable(GL_DEPTH_TEST);
	glEnable(GL_CULL_FACE);

	glMatrixMode(GL_MODELVIEW);
	glLoadMatrixf(&ViewMatrix);

	if(WireFrame)
	{
		glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
	}

	glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_CUBE_MAP, PoolSkyCubeMap);
	glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_CUBE_MAP, PhotonsCubeMap);
	glUseProgram(PoolSkyProgram);
	glBindBuffer(GL_ARRAY_BUFFER, PoolSkyVBO);
	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(3, GL_FLOAT, 12, (void*)0);
	glDrawArrays(GL_QUADS, 0, 24);
	glDisableClientState(GL_VERTEX_ARRAY);
	glBindBuffer(GL_ARRAY_BUFFER, 0);
	glUseProgram(0);
	glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_CUBE_MAP, 0);
	glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_CUBE_MAP, 0);

	glDisable(GL_CULL_FACE);

	// render water surface ---------------------------------------------------------------------------------------------------

	glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, WaterHeightMaps[WHMID]);
	glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_2D, WaterNormalMap);
	glActiveTexture(GL_TEXTURE2); glBindTexture(GL_TEXTURE_CUBE_MAP, PoolSkyCubeMap);
	glActiveTexture(GL_TEXTURE3); glBindTexture(GL_TEXTURE_CUBE_MAP, PhotonsCubeMap);
	glUseProgram(WaterProgram);
	glUniform3fv(WaterProgram.UniformLocations[0], 1, &Camera.Position);
	glBindBuffer(GL_ARRAY_BUFFER, WaterVBO);
	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(3, GL_FLOAT, 12, (void*)0);
	glDrawArrays(GL_QUADS, 0, QuadsVerticesCount);
	glDisableClientState(GL_VERTEX_ARRAY);
	glBindBuffer(GL_ARRAY_BUFFER, 0);
	glUseProgram(0);
	glActiveTexture(GL_TEXTURE3); glBindTexture(GL_TEXTURE_CUBE_MAP, 0);
	glActiveTexture(GL_TEXTURE2); glBindTexture(GL_TEXTURE_CUBE_MAP, 0);
	glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_2D, 0);
	glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, 0);

	if(WireFrame)
	{
		glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
	}

	glDisable(GL_DEPTH_TEST);
}

void COpenGLRenderer::Resize(int Width, int Height)
{
	this->Width = Width;
	this->Height = Height;

	ProjectionMatrix = perspective(45.0f, (float)Width / (float)Height, 0.125f, 512.0f);
	ProjectionBiasMatrixInverse = inverse(ProjectionMatrix) * BiasMatrixInverse;
}

void COpenGLRenderer::Destroy()
{
	PoolSkyCubeMap.Destroy();

	WaterAddDropProgram.Destroy();
	WaterHeightMapProgram.Destroy();
	WaterNormalMapProgram.Destroy();
	PhotonProgram.Destroy();
	CubeMapHBlurProgram.Destroy();
	CubeMapVBlurProgram.Destroy();
	PoolSkyProgram.Destroy();
	WaterProgram.Destroy();

	glDeleteTextures(2, WaterHeightMaps);
	glDeleteTextures(1, &WaterNormalMap);
	glDeleteTextures(1, &PhotonsTexture);
	glDeleteTextures(2, PhotonsTempCubeMaps);
	glDeleteTextures(1, &PhotonsCubeMap);

	glDeleteBuffers(1, &PhotonsVBO);
	glDeleteBuffers(1, &PoolSkyVBO);
	glDeleteBuffers(1, &WaterVBO);

	if(GLEW_EXT_framebuffer_object)
	{
		glDeleteFramebuffersEXT(1, &FBO);
	}
}

void COpenGLRenderer::AddDrop(float x, float y, float DropRadius)
{
	if(x >= -1.0f && x <= 1.0f && y >= -1.0f && y <= 1.0f)
	{
		glViewport(0, 0, WMR, WMR);

		glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, FBO);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_2D, WaterHeightMaps[(WHMID + 1) % 2], 0);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_TEXTURE_2D, 0, 0);

		glBindTexture(GL_TEXTURE_2D, WaterHeightMaps[WHMID]);
		glUseProgram(WaterAddDropProgram);
		glUniform1f(WaterAddDropProgram.UniformLocations[0], DropRadius);
		glUniform2fv(WaterAddDropProgram.UniformLocations[1], 1, &vec2(x * 0.5f + 0.5f, 0.5f - y * 0.5f));
		glBegin(GL_QUADS);
			glVertex2f(0.0f, 0.0f);
			glVertex2f(1.0f, 0.0f);
			glVertex2f(1.0f, 1.0f);
			glVertex2f(0.0f, 1.0f);
		glEnd();
		glUseProgram(0);
		glBindTexture(GL_TEXTURE_2D, 0);

		glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);

		++WHMID %= 2;
	}
}

void COpenGLRenderer::AddDropByMouseClick(int x, int y)
{
	float s = (float)x / (float)(Width - 1);
	float t = 1.0f - (float)y / (float)(Height - 1);

	vec4 Position = ViewMatrixInverse * (ProjectionBiasMatrixInverse * vec4(s, t, 0.5f, 1.0f));
	Position /= Position.w;

	vec3 Ray = normalize(*(vec3*)&Position - Camera.Position);

	vec3 Normal = vec3(0.0f, 1.0f, 0.0f);
	float D = -dot(Normal, vec3(0.0f, 0.0f, 0.0f));

	float NdotR = -dot(Normal, Ray);

	if(NdotR != 0.0f)
	{
		float Distance = (dot(Normal, Camera.Position) + D) / NdotR;

		if(Distance > 0.0f)
		{
			vec3 Position = Ray * Distance + Camera.Position;

			AddDrop(Position.x, Position.z, DropRadius);
		}
	}
}

// ----------------------------------------------------------------------------------------------------------------------------

COpenGLRenderer OpenGLRenderer;

// ----------------------------------------------------------------------------------------------------------------------------

CString ModuleDirectory, ErrorLog;

// ----------------------------------------------------------------------------------------------------------------------------

void GetModuleDirectory()
{
	char *moduledirectory = new char[256];
	GetModuleFileName(GetModuleHandle(NULL), moduledirectory, 256);
	*(strrchr(moduledirectory, '\\') + 1) = 0;
	ModuleDirectory = moduledirectory;
	delete [] moduledirectory;
}

// ----------------------------------------------------------------------------------------------------------------------------

COpenGLView::COpenGLView()
{
}

COpenGLView::~COpenGLView()
{
}

bool COpenGLView::Init(HINSTANCE hInstance, char *Title, int Width, int Height, int Samples)
{
	this->Title = Title;
	this->Width = Width;
	this->Height = Height;

	WNDCLASSEX WndClassEx;

	memset(&WndClassEx, 0, sizeof(WNDCLASSEX));

	WndClassEx.cbSize = sizeof(WNDCLASSEX);
	WndClassEx.style = CS_OWNDC | CS_HREDRAW | CS_VREDRAW;
	WndClassEx.lpfnWndProc = WndProc;
	WndClassEx.hInstance = hInstance;
	WndClassEx.hIcon = LoadIcon(NULL, IDI_APPLICATION);
	WndClassEx.hIconSm = LoadIcon(NULL, IDI_APPLICATION);
	WndClassEx.hCursor = LoadCursor(NULL, IDC_ARROW);
	WndClassEx.lpszClassName = "Win32OpenGLWindowClass";

	if(RegisterClassEx(&WndClassEx) == 0)
	{
		ErrorLog.Set("RegisterClassEx failed!");
		return false;
	}

	DWORD Style = WS_OVERLAPPEDWINDOW | WS_CLIPSIBLINGS | WS_CLIPCHILDREN;

	hWnd = CreateWindowEx(WS_EX_APPWINDOW, WndClassEx.lpszClassName, Title, Style, 0, 0, Width, Height, NULL, NULL, hInstance, NULL);

	if(hWnd == NULL)
	{
		ErrorLog.Set("CreateWindowEx failed!");
		return false;
	}

	HDC hDC = GetDC(hWnd);

	if(hDC == NULL)
	{
		ErrorLog.Set("GetDC failed!");
		return false;
	}

	PIXELFORMATDESCRIPTOR pfd;

	memset(&pfd, 0, sizeof(PIXELFORMATDESCRIPTOR));

	pfd.nSize = sizeof(PIXELFORMATDESCRIPTOR);
	pfd.nVersion = 1;
	pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
	pfd.iPixelType = PFD_TYPE_RGBA;
	pfd.cColorBits = 32;
	pfd.cDepthBits = 24;
	pfd.iLayerType = PFD_MAIN_PLANE;

	int PixelFormat = ChoosePixelFormat(hDC, &pfd);

	if(PixelFormat == 0)
	{
		ErrorLog.Set("ChoosePixelFormat failed!");
		return false;
	}

	static int MSAAPixelFormat = 0;

	if(SetPixelFormat(hDC, MSAAPixelFormat == 0 ? PixelFormat : MSAAPixelFormat, &pfd) == FALSE)
	{
		ErrorLog.Set("SetPixelFormat failed!");
		return false;
	}

	hGLRC = wglCreateContext(hDC);

	if(hGLRC == NULL)
	{
		ErrorLog.Set("wglCreateContext failed!");
		return false;
	}

	if(wglMakeCurrent(hDC, hGLRC) == FALSE)
	{
		ErrorLog.Set("wglMakeCurrent failed!");
		return false;
	}

	if(glewInit() != GLEW_OK)
	{
		ErrorLog.Set("glewInit failed!");
		return false;
	}

	if(!GLEW_VERSION_2_1)
	{
		ErrorLog.Set("OpenGL 2.1 not supported!");
		return false;
	}

	if(MSAAPixelFormat == 0 && Samples > 0)
	{
		if(GLEW_ARB_multisample && WGLEW_ARB_pixel_format)
		{
			while(Samples > 0)
			{
				UINT NumFormats = 0;

				int PFAttribs[] =
				{
					WGL_DRAW_TO_WINDOW_ARB, GL_TRUE,
					WGL_SUPPORT_OPENGL_ARB, GL_TRUE,
					WGL_DOUBLE_BUFFER_ARB, GL_TRUE,
					WGL_PIXEL_TYPE_ARB, WGL_TYPE_RGBA_ARB,
					WGL_COLOR_BITS_ARB, 32,
					WGL_DEPTH_BITS_ARB, 24,
					WGL_ACCELERATION_ARB, WGL_FULL_ACCELERATION_ARB,
					WGL_SAMPLE_BUFFERS_ARB, GL_TRUE,
					WGL_SAMPLES_ARB, Samples,
					0
				};

				if(wglChoosePixelFormatARB(hDC, PFAttribs, NULL, 1, &MSAAPixelFormat, &NumFormats) == TRUE && NumFormats > 0) break;

				Samples--;
			}

			wglDeleteContext(hGLRC);
			DestroyWindow(hWnd);
			UnregisterClass(WndClassEx.lpszClassName, hInstance);

			return Init(hInstance, Title, Width, Height, Samples);
		}
		else
		{
			Samples = 0;
		}
	}

	this->Samples = Samples;

	GetModuleDirectory();

	glGetIntegerv(GL_MAX_TEXTURE_SIZE, &gl_max_texture_size);

	if(GLEW_EXT_texture_filter_anisotropic)
	{
		glGetIntegerv(GL_MAX_TEXTURE_MAX_ANISOTROPY_EXT, &gl_max_texture_max_anisotropy_ext);
	}

	if(WGLEW_EXT_swap_control)
	{
		wglSwapIntervalEXT(0);
	}

	return OpenGLRenderer.Init();
}

void COpenGLView::Show(bool Maximized)
{
	RECT dRect, wRect, cRect;

	GetWindowRect(GetDesktopWindow(), &dRect);
	GetWindowRect(hWnd, &wRect);
	GetClientRect(hWnd, &cRect);

	wRect.right += Width - cRect.right;
	wRect.bottom += Height - cRect.bottom;
	wRect.right -= wRect.left;
	wRect.bottom -= wRect.top;
	wRect.left = dRect.right / 2 - wRect.right / 2;
	wRect.top = dRect.bottom / 2 - wRect.bottom / 2;

	MoveWindow(hWnd, wRect.left, wRect.top, wRect.right, wRect.bottom, FALSE);

	ShowWindow(hWnd, Maximized ? SW_SHOWMAXIMIZED : SW_SHOWNORMAL);
}

void COpenGLView::MessageLoop()
{
	MSG Msg;

	while(GetMessage(&Msg, NULL, 0, 0) > 0)
	{
		TranslateMessage(&Msg);
		DispatchMessage(&Msg);
	}
}

void COpenGLView::Destroy()
{
	if(GLEW_VERSION_2_1)
	{
		OpenGLRenderer.Destroy();
	}

	wglDeleteContext(hGLRC);
	DestroyWindow(hWnd);
}

void COpenGLView::OnKeyDown(UINT Key)
{
	switch(Key)
	{
		case VK_F1:
			OpenGLRenderer.WireFrame = !OpenGLRenderer.WireFrame;
			break;

		case '1':
			OpenGLRenderer.DropRadius = 4.0f / 256.0f;
			break;

		case '2':
			OpenGLRenderer.DropRadius = 4.0f / 128.0f;
			break;

		case '3':
			OpenGLRenderer.DropRadius = 4.0f / 64.0f;
			break;

		case '4':
			OpenGLRenderer.DropRadius = 4.0f / 32.0f;
			break;

		case '5':
			OpenGLRenderer.DropRadius = 4.0f / 16.0f;
			break;

		case VK_SPACE:
			OpenGLRenderer.Pause = !OpenGLRenderer.Pause;
			break;
	}
}

void COpenGLView::OnLButtonDown(int X, int Y)
{
	OpenGLRenderer.AddDropByMouseClick(X, Y);
}

void COpenGLView::OnMouseMove(int X, int Y)
{
	if(GetKeyState(VK_RBUTTON) & 0x80)
	{
		Camera.OnMouseMove(LastX - X, LastY - Y);

		LastX = X;
		LastY = Y;
	}
}

void COpenGLView::OnMouseWheel(short zDelta)
{
	Camera.OnMouseWheel(zDelta);
}

void COpenGLView::OnPaint()
{
	static DWORD LastFPSTime = GetTickCount(), LastFrameTime = LastFPSTime, FPS = 0;

	PAINTSTRUCT ps;

	HDC hDC = BeginPaint(hWnd, &ps);

	DWORD Time = GetTickCount();

	float FrameTime = (Time - LastFrameTime) * 0.001f;

	LastFrameTime = Time;

	if(Time - LastFPSTime > 1000)
	{
		CString Text = Title;

		if(OpenGLRenderer.Text[0] != 0)
		{
			Text.Append(" - " + OpenGLRenderer.Text);
		}

		Text.Append(" - %dx%d", Width, Height);
		Text.Append(", ATF %dx", gl_max_texture_max_anisotropy_ext);
		Text.Append(", MSAA %dx", Samples);
		Text.Append(", FPS: %d", FPS);
		Text.Append(" - %s", glGetString(GL_RENDERER));
		
		SetWindowText(hWnd, Text);

		LastFPSTime = Time;
		FPS = 0;
	}
	else
	{
		FPS++;
	}

	BYTE Keys = 0x00;

	if(GetKeyState('W') & 0x80) Keys |= 0x01;
	if(GetKeyState('S') & 0x80) Keys |= 0x02;
	if(GetKeyState('A') & 0x80) Keys |= 0x04;
	if(GetKeyState('D') & 0x80) Keys |= 0x08;
	if(GetKeyState('R') & 0x80) Keys |= 0x10;
	if(GetKeyState('F') & 0x80) Keys |= 0x20;

	if(GetKeyState(VK_SHIFT) & 0x80) Keys |= 0x40;
	if(GetKeyState(VK_CONTROL) & 0x80) Keys |= 0x80;

	if(Keys & 0x3F)
	{
		Camera.Move(Camera.OnKeys(Keys, FrameTime));
	}

	OpenGLRenderer.Render(FrameTime);

	SwapBuffers(hDC);

	EndPaint(hWnd, &ps);

	InvalidateRect(hWnd, NULL, FALSE);
}

void COpenGLView::OnRButtonDown(int X, int Y)
{
	LastX = X;
	LastY = Y;
}

void COpenGLView::OnSize(int Width, int Height)
{
	this->Width = Width;
	this->Height = Height;

	OpenGLRenderer.Resize(Width, Height);
}

// ----------------------------------------------------------------------------------------------------------------------------

COpenGLView OpenGLView;

// ----------------------------------------------------------------------------------------------------------------------------

LRESULT CALLBACK WndProc(HWND hWnd, UINT uiMsg, WPARAM wParam, LPARAM lParam)
{
	switch(uiMsg)
	{
		case WM_CLOSE:
			PostQuitMessage(0);
			break;

		case WM_KEYDOWN:
			OpenGLView.OnKeyDown((UINT)wParam);
			break;

		case WM_LBUTTONDOWN:
			OpenGLView.OnLButtonDown(LOWORD(lParam), HIWORD(lParam));
			break;

		case WM_MOUSEMOVE:
			OpenGLView.OnMouseMove(LOWORD(lParam), HIWORD(lParam));
			break;

		case 0x020A: // WM_MOUSWHEEL
			OpenGLView.OnMouseWheel(HIWORD(wParam));
			break;

		case WM_PAINT:
			OpenGLView.OnPaint();
			break;

		case WM_RBUTTONDOWN:
			OpenGLView.OnRButtonDown(LOWORD(lParam), HIWORD(lParam));
			break;

		case WM_SIZE:
			OpenGLView.OnSize(LOWORD(lParam), HIWORD(lParam));
			break;

		default:
			return DefWindowProc(hWnd, uiMsg, wParam, lParam);
	}

	return 0;
}

// ----------------------------------------------------------------------------------------------------------------------------

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR sCmdLine, int iShow)
{
	char *AppName = "Water waves caustic GPU algorithm";

	if(OpenGLView.Init(hInstance, AppName, 800, 600, 0))
	{
		OpenGLView.Show();
		OpenGLView.MessageLoop();
	}
	else
	{
		MessageBox(NULL, ErrorLog, AppName, MB_OK | MB_ICONERROR);
	}

	OpenGLView.Destroy();

	return 0;
}
