#include "opengl_21_tutorials_win32_framework.h"

// ----------------------------------------------------------------------------------------------------------------------------

CBuffer::CBuffer()
{
	SetDefaults();
}

CBuffer::~CBuffer()
{
	Empty();
}

void CBuffer::AddData(void *Data, int DataSize)
{
	int Remaining = BufferSize - Position;

	if(DataSize > Remaining)
	{
		BYTE *OldBuffer = Buffer;
		int OldBufferSize = BufferSize;

		int Needed = DataSize - Remaining;

		BufferSize += Needed > BUFFER_SIZE_INCREMENT ? Needed : BUFFER_SIZE_INCREMENT;

		Buffer = new BYTE[BufferSize];

		memcpy(Buffer, OldBuffer, OldBufferSize);

		delete [] OldBuffer;
	}

	memcpy(Buffer + Position, Data, DataSize);

	Position += DataSize;
}

void CBuffer::Empty()
{
	delete [] Buffer;

	SetDefaults();
}

void *CBuffer::GetData()
{
	return Buffer;
}

int CBuffer::GetDataSize()
{
	return Position;
}

void CBuffer::SetDefaults()
{
	Buffer = NULL;

	BufferSize = 0;
	Position = 0;
}

// ----------------------------------------------------------------------------------------------------------------------------

int gl_max_texture_size = 0, gl_max_texture_max_anisotropy_ext = 0;

// ----------------------------------------------------------------------------------------------------------------------------

CTexture::CTexture()
{
	Texture = 0;
}

CTexture::~CTexture()
{
}

CTexture::operator GLuint ()
{
	return Texture;
}

bool CTexture::LoadTexture2D(char *FileName)
{
	CString DirectoryFileName = ModuleDirectory + FileName;

	int Width, Height, BPP;

	FIBITMAP *dib = GetBitmap(DirectoryFileName, Width, Height, BPP);

	if(dib == NULL)
	{
		ErrorLog.Append("Error loading texture " + DirectoryFileName + "!\r\n");
		return false;
	}

	GLenum Format = 0;

	if(BPP == 32) Format = GL_BGRA;
	if(BPP == 24) Format = GL_BGR;

	if(Format == 0)
	{
		ErrorLog.Append("Unsupported texture format (%s)!\r\n", FileName);
		FreeImage_Unload(dib);
		return false;
	}

	Destroy();

	glGenTextures(1, &Texture);

	glBindTexture(GL_TEXTURE_2D, Texture);

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	if(GLEW_EXT_texture_filter_anisotropic)
	{
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY_EXT, gl_max_texture_max_anisotropy_ext);
	}

	glTexParameteri(GL_TEXTURE_2D, GL_GENERATE_MIPMAP, GL_TRUE);

	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, Width, Height, 0, Format, GL_UNSIGNED_BYTE, FreeImage_GetBits(dib));

	glBindTexture(GL_TEXTURE_2D, 0);

	FreeImage_Unload(dib);

	return true;
}

bool CTexture::LoadTextureCubeMap(char **FileNames)
{
	int Width, Height, BPP;

	FIBITMAP *dib[6];

	bool Error = false;
	
	for(int i = 0; i < 6; i++)
	{
		CString DirectoryFileName = ModuleDirectory + FileNames[i];

		dib[i] = GetBitmap(DirectoryFileName, Width, Height, BPP);

		if(dib[i] == NULL)
		{
			ErrorLog.Append("Error loading texture " + DirectoryFileName + "!\r\n");
			Error = true;
		}
	}

	if(Error)
	{
		for(int i = 0; i < 6; i++)
		{
			FreeImage_Unload(dib[i]);
		}

		return false;
	}

	GLenum Format = 0;
	
	if(BPP == 32) Format = GL_BGRA;
	if(BPP == 24) Format = GL_BGR;

	if(Format == 0)
	{
		ErrorLog.Append("Unsupported texture format (%s)!\r\n", FileNames[5]);

		for(int i = 0; i < 6; i++)
		{
			FreeImage_Unload(dib[i]);
		}

		return false;
	}

	Destroy();

	glGenTextures(1, &Texture);

	glBindTexture(GL_TEXTURE_CUBE_MAP, Texture);

	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	if(GLEW_EXT_texture_filter_anisotropic)
	{
		glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MAX_ANISOTROPY_EXT, gl_max_texture_max_anisotropy_ext);
	}

	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_GENERATE_MIPMAP, GL_TRUE);

	for(int i = 0; i < 6; i++)
	{
		glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_X + i, 0, GL_RGBA8, Width, Height, 0, Format, GL_UNSIGNED_BYTE, FreeImage_GetBits(dib[i]));
	}

	glBindTexture(GL_TEXTURE_CUBE_MAP, 0);

	for(int i = 0; i < 6; i++)
	{
		FreeImage_Unload(dib[i]);
	}

	return true;
}

void CTexture::Destroy()
{
	glDeleteTextures(1, &Texture);
	Texture = 0;
}

FIBITMAP *CTexture::GetBitmap(char *FileName, int &Width, int &Height, int &BPP)
{
	FREE_IMAGE_FORMAT fif = FreeImage_GetFileType(FileName);

	if(fif == FIF_UNKNOWN)
	{
		fif = FreeImage_GetFIFFromFilename(FileName);
	}

	if(fif == FIF_UNKNOWN)
	{
		return NULL;
	}

	FIBITMAP *dib = NULL;

	if(FreeImage_FIFSupportsReading(fif))
	{
		dib = FreeImage_Load(fif, FileName);
	}

	if(dib != NULL)
	{
		int OriginalWidth = FreeImage_GetWidth(dib);
		int OriginalHeight = FreeImage_GetHeight(dib);

		Width = OriginalWidth;
		Height = OriginalHeight;

		if(Width == 0 || Height == 0)
		{
			FreeImage_Unload(dib);
			return NULL;
		}

		BPP = FreeImage_GetBPP(dib);

		if(Width > gl_max_texture_size) Width = gl_max_texture_size;
		if(Height > gl_max_texture_size) Height = gl_max_texture_size;

		if(!GLEW_ARB_texture_non_power_of_two)
		{
			Width = 1 << (int)floor((log((float)Width) / log(2.0f)) + 0.5f); 
			Height = 1 << (int)floor((log((float)Height) / log(2.0f)) + 0.5f);
		}

		if(Width != OriginalWidth || Height != OriginalHeight)
		{
			FIBITMAP *rdib = FreeImage_Rescale(dib, Width, Height, FILTER_BICUBIC);
			FreeImage_Unload(dib);
			dib = rdib;
		}
	}

	return dib;
}

// ----------------------------------------------------------------------------------------------------------------------------

CShaderProgram::CShaderProgram()
{
	SetDefaults();
}

CShaderProgram::~CShaderProgram()
{
}

CShaderProgram::operator GLuint ()
{
	return Program;
}

bool CShaderProgram::Load(char *VertexShaderFileName, char *FragmentShaderFileName)
{
	bool Error = false;

	Destroy();

	Error |= ((VertexShader = LoadShader(VertexShaderFileName, GL_VERTEX_SHADER)) == 0);
	Error |= ((FragmentShader = LoadShader(FragmentShaderFileName, GL_FRAGMENT_SHADER)) == 0);

	if(Error)
	{
		Destroy();
		return false;
	}

	Program = glCreateProgram();
	glAttachShader(Program, VertexShader);
	glAttachShader(Program, FragmentShader);
	glLinkProgram(Program);

	int LinkStatus;
	glGetProgramiv(Program, GL_LINK_STATUS, &LinkStatus);

	if(LinkStatus == GL_FALSE)
	{
		ErrorLog.Append("Error linking program (%s, %s)!\r\n", VertexShaderFileName, FragmentShaderFileName);

		int InfoLogLength = 0;
		glGetProgramiv(Program, GL_INFO_LOG_LENGTH, &InfoLogLength);
	
		if(InfoLogLength > 0)
		{
			char *InfoLog = new char[InfoLogLength];
			int CharsWritten  = 0;
			glGetProgramInfoLog(Program, InfoLogLength, &CharsWritten, InfoLog);
			ErrorLog.Append(InfoLog);
			delete [] InfoLog;
		}

		Destroy();

		return false;
	}

	return true;
}

void CShaderProgram::Destroy()
{
	glDetachShader(Program, VertexShader);
	glDetachShader(Program, FragmentShader);

	glDeleteShader(VertexShader);
	glDeleteShader(FragmentShader);

	glDeleteProgram(Program);

	delete [] UniformLocations;
	delete [] AttribLocations;

	SetDefaults();
}

GLuint CShaderProgram::LoadShader(char *FileName, GLenum Type)
{
	CString DirectoryFileName = ModuleDirectory + FileName;

	FILE *File;

	if(fopen_s(&File, DirectoryFileName, "rb") != 0)
	{
		ErrorLog.Append("Error loading file " + DirectoryFileName + "!\r\n");
		return 0;
	}

	fseek(File, 0, SEEK_END);
	long Size = ftell(File);
	fseek(File, 0, SEEK_SET);
	char *Source = new char[Size + 1];
	fread(Source, 1, Size, File);
	fclose(File);
	Source[Size] = 0;

	GLuint Shader = glCreateShader(Type);

	glShaderSource(Shader, 1, (const char**)&Source, NULL);
	delete [] Source;
	glCompileShader(Shader);

	int CompileStatus;
	glGetShaderiv(Shader, GL_COMPILE_STATUS, &CompileStatus);

	if(CompileStatus == GL_FALSE)
	{
		ErrorLog.Append("Error compiling shader %s!\r\n", FileName);

		int InfoLogLength = 0;
		glGetShaderiv(Shader, GL_INFO_LOG_LENGTH, &InfoLogLength);
	
		if(InfoLogLength > 0)
		{
			char *InfoLog = new char[InfoLogLength];
			int CharsWritten  = 0;
			glGetShaderInfoLog(Shader, InfoLogLength, &CharsWritten, InfoLog);
			ErrorLog.Append(InfoLog);
			delete [] InfoLog;
		}

		glDeleteShader(Shader);

		return 0;
	}

	return Shader;
}

void CShaderProgram::SetDefaults()
{
	VertexShader = 0;
	FragmentShader = 0;

	Program = 0;

	UniformLocations = NULL;
	AttribLocations = NULL;
}

// ----------------------------------------------------------------------------------------------------------------------------

CCamera::CCamera()
{
	ViewMatrix = NULL;
	ViewMatrixInverse = NULL;

	X = vec3(1.0f, 0.0f, 0.0f);
	Y = vec3(0.0f, 1.0f, 0.0f);
	Z = vec3(0.0f, 0.0f, 1.0f);

	Position = vec3(0.0f, 0.0f, 5.0f);
	Reference = vec3(0.0f, 0.0f, 0.0f);
}

CCamera::~CCamera()
{
}

void CCamera::Look(const vec3 &Position, const vec3 &Reference, bool RotateAroundReference)
{
	this->Position = Position;
	this->Reference = Reference;

	Z = normalize(Position - Reference);
	X = normalize(cross(vec3(0.0f, 1.0f, 0.0f), Z));
	Y = cross(Z, X);

	if(!RotateAroundReference)
	{
		this->Reference = this->Position;
		this->Position += Z * 0.05f;
	}

	CalculateViewMatrix();
}

void CCamera::Move(const vec3 &Movement)
{
	Position += Movement;
	Reference += Movement;

	CalculateViewMatrix();
}

vec3 CCamera::OnKeys(BYTE Keys, float FrameTime)
{
	float Speed = 5.0f;

	if(Keys & 0x40) Speed *= 2.0f;
	if(Keys & 0x80) Speed *= 0.5f;

	float Distance = Speed * FrameTime;

	vec3 Up(0.0f, 1.0f, 0.0f);
	vec3 Right = X;
	vec3 Forward = cross(Up, Right);

	Up *= Distance;
	Right *= Distance;
	Forward *= Distance;

	vec3 Movement;

	if(Keys & 0x01) Movement += Forward;
	if(Keys & 0x02) Movement -= Forward;
	if(Keys & 0x04) Movement -= Right;
	if(Keys & 0x08) Movement += Right;
	if(Keys & 0x10) Movement += Up;
	if(Keys & 0x20) Movement -= Up;

	return Movement;
}

void CCamera::OnMouseMove(int dx, int dy)
{
	float Sensitivity = 0.25f;

	Position -= Reference;

	if(dx != 0)
	{
		float DeltaX = (float)dx * Sensitivity;

		X = rotate(X, DeltaX, vec3(0.0f, 1.0f, 0.0f));
		Y = rotate(Y, DeltaX, vec3(0.0f, 1.0f, 0.0f));
		Z = rotate(Z, DeltaX, vec3(0.0f, 1.0f, 0.0f));
	}

	if(dy != 0)
	{
		float DeltaY = (float)dy * Sensitivity;

		Y = rotate(Y, DeltaY, X);
		Z = rotate(Z, DeltaY, X);

		if(Y.y < 0.0f)
		{
			Z = vec3(0.0f, Z.y > 0.0f ? 1.0f : -1.0f, 0.0f);
			Y = cross(Z, X);
		}
	}

	Position = Reference + Z * length(Position);

	CalculateViewMatrix();
}

void CCamera::OnMouseWheel(float zDelta)
{
	Position -= Reference;

	if(zDelta < 0 && length(Position) < 500.0f)
	{
		Position += Position * 0.1f;
	}

	if(zDelta > 0 && length(Position) > 0.05f)
	{
		Position -= Position * 0.1f;
	}

	Position += Reference;

	CalculateViewMatrix();
}

void CCamera::SetViewMatrixPointer(float *ViewMatrix, float *ViewMatrixInverse)
{
	this->ViewMatrix = (mat4x4*)ViewMatrix;
	this->ViewMatrixInverse = (mat4x4*)ViewMatrixInverse;

	CalculateViewMatrix();
}

void CCamera::CalculateViewMatrix()
{
	if(ViewMatrix != NULL)
	{
		*ViewMatrix = mat4x4(X.x, Y.x, Z.x, 0.0f, X.y, Y.y, Z.y, 0.0f, X.z, Y.z, Z.z, 0.0f, -dot(X, Position), -dot(Y, Position), -dot(Z, Position), 1.0f);

		if(ViewMatrixInverse != NULL)
		{
			*ViewMatrixInverse = inverse(*ViewMatrix);
		}
	}
}

// ----------------------------------------------------------------------------------------------------------------------------

CCamera Camera;

// ----------------------------------------------------------------------------------------------------------------------------

CTriangle::CTriangle()
{
}

CTriangle::CTriangle(const vec3 &a, const vec3 &b, const vec3 &c, const vec3 &Color) : a(a), b(b), c(c), Color(Color)
{
	ab = b - a; bc = c - b; ca = a - c;

	N = normalize(cross(ab, -ca));
	D = -dot(N, a);

	N1 = normalize(cross(N, ab));
	D1 = -dot(N1, a);

	N2 = normalize(cross(N, bc));
	D2 = -dot(N2, b);

	N3 = normalize(cross(N, ca));
	D3 = -dot(N3, c);

	lab = length(ab); ab /= lab;
	lbc = length(bc); bc /= lbc;
	lca = length(ca); ca /= lca;
}

bool CTriangle::Inside(float x, float y, float z)
{
	if(N1.x * x + N1.y * y + N1.z * z + D1 < 0.0f) return false;
	if(N2.x * x + N2.y * y + N2.z * z + D2 < 0.0f) return false;
	if(N3.x * x + N3.y * y + N3.z * z + D3 < 0.0f) return false;

	return true;
}

bool CTriangle::Inside(const vec3 &Point)
{
	if(dot(N1, Point) + D1 < 0.0f) return false;
	if(dot(N2, Point) + D2 < 0.0f) return false;
	if(dot(N3, Point) + D3 < 0.0f) return false;

	return true;
}

bool CTriangle::Intersect(const vec3 &Origin, const vec3 &Ray, float MaxDistance, float &Distance, vec3 &Point)
{
	float NdotR = -dot(N, Ray);

	if(NdotR > 0.0f)
	{
		Distance = (dot(N, Origin) + D) / NdotR;

		if(Distance >= 0.0f && Distance < MaxDistance)
		{
			Point = Ray * Distance + Origin;

			return Inside(Point);
		}
	}

	return false;
}

bool CTriangle::Intersect(const vec3 &Origin, const vec3 &Ray, float MaxDistance, float &Distance)
{
	float NdotR = -dot(N, Ray);

	if(NdotR > 0.0f)
	{
		Distance = (dot(N, Origin) + D) / NdotR;

		if(Distance >= 0.0f && Distance < MaxDistance)
		{
			return Inside(Ray * Distance + Origin);
		}
	}

	return false;
}

bool CTriangle::Intersect(const vec3 &Origin, const vec3 &Ray, float MaxDistance)
{
	float NdotR = -dot(N, Ray);

	if(NdotR > 0.0f)
	{
		float Distance = (dot(N, Origin) + D) / NdotR;

		if(Distance >= 0.0f && Distance < MaxDistance)
		{
			return Inside(Ray * Distance + Origin);
		}
	}

	return false;
}

// ----------------------------------------------------------------------------------------------------------------------------

RTData::RTData()
{
	Distance = 1048576.0f;
	Triangle = NULL;
}

// ----------------------------------------------------------------------------------------------------------------------------

CVoxel::CVoxel()
{
	Triangles = NULL;
	TrianglesCount = 0;
	MaxTrianglesCount = 1;
	Size = 0.0f;
}

CVoxel::~CVoxel()
{
}

void CVoxel::Add(CTriangle *Triangle)
{
	if(TrianglesCount % MaxTrianglesCount == 0)
	{
		CTriangle **OldTriangles = Triangles;

		MaxTrianglesCount *= 2;

		Triangles = new CTriangle*[MaxTrianglesCount];

		for(int i = 0; i < TrianglesCount; i++)
		{
			Triangles[i] = OldTriangles[i];
		}

		if(OldTriangles != NULL)
		{
			delete [] OldTriangles;
		}
	}

	Triangles[TrianglesCount] = Triangle;

	TrianglesCount++;
}

void CVoxel::Delete()
{
	if(Triangles != NULL)
	{
		delete [] Triangles;
		Triangles = NULL;
		TrianglesCount = 0;
		MaxTrianglesCount = 1;
		Size = 0.0f;
		Min = Max = MinE = MaxE = vec3(0.0f);
	}
}

bool CVoxel::Inside(const vec3 &Point)
{
	if(MinE.x < Point.x && Point.x < MaxE.x)
	{
		if(MinE.y < Point.y && Point.y < MaxE.y)
		{
			if(MinE.z < Point.z && Point.z < MaxE.z)
			{
				return true;
			}
		}
	}

	return false;
}

bool CVoxel::IntersectEdgesX(CTriangle *Triangle, float x, float y1, float y2, float z1, float z2)
{
	float NdotR = -Triangle->N.x;

	if(NdotR != 0.0f)
	{
		vec3 Origin = vec3(x, y1, z1);

		float Distance = (dot(Triangle->N, Origin) + Triangle->D) / NdotR;

		if(Distance >= 0.0f && Distance <= Size)
		{
			return Triangle->Inside(Origin.x + Distance, Origin.y, Origin.z);
		}

		Origin.z = z2;

		Distance = (dot(Triangle->N, Origin) + Triangle->D) / NdotR;

		if(Distance >= 0.0f && Distance <= Size)
		{
			return Triangle->Inside(Origin.x + Distance, Origin.y, Origin.z);
		}

		Origin.y = y2;

		Distance = (dot(Triangle->N, Origin) + Triangle->D) / NdotR;

		if(Distance >= 0.0f && Distance <= Size)
		{
			return Triangle->Inside(Origin.x + Distance, Origin.y, Origin.z);
		}

		Origin.z = z1;

		Distance = (dot(Triangle->N, Origin) + Triangle->D) / NdotR;

		if(Distance >= 0.0f && Distance <= Size)
		{
			return Triangle->Inside(Origin.x + Distance, Origin.y, Origin.z);
		}
	}

	return false;
}

bool CVoxel::IntersectEdgesY(CTriangle *Triangle, float y, float x1, float x2, float z1, float z2)
{
	float NdotR = -Triangle->N.y;

	if(NdotR != 0.0f)
	{
		vec3 Origin = vec3(x1, y, z1);

		float Distance = (dot(Triangle->N, Origin) + Triangle->D) / NdotR;

		if(Distance >= 0.0f && Distance <= Size)
		{
			return Triangle->Inside(Origin.x, Origin.y + Distance, Origin.z);
		}

		Origin.x = x2;

		Distance = (dot(Triangle->N, Origin) + Triangle->D) / NdotR;

		if(Distance >= 0.0f && Distance <= Size)
		{
			return Triangle->Inside(Origin.x, Origin.y + Distance, Origin.z);
		}

		Origin.z = z2;

		Distance = (dot(Triangle->N, Origin) + Triangle->D) / NdotR;

		if(Distance >= 0.0f && Distance <= Size)
		{
			return Triangle->Inside(Origin.x, Origin.y + Distance, Origin.z);
		}

		Origin.x = x1;

		Distance = (dot(Triangle->N, Origin) + Triangle->D) / NdotR;

		if(Distance >= 0.0f && Distance <= Size)
		{
			return Triangle->Inside(Origin.x, Origin.y + Distance, Origin.z);
		}
	}

	return false;
}

bool CVoxel::IntersectEdgesZ(CTriangle *Triangle, float z, float x1, float x2, float y1, float y2)
{
	float NdotR = -Triangle->N.z;

	if(NdotR != 0.0f)
	{
		vec3 Origin = vec3(x1, y1, z);

		float Distance = (dot(Triangle->N, Origin) + Triangle->D) / NdotR;

		if(Distance >= 0.0f && Distance <= Size)
		{
			return Triangle->Inside(Origin.x, Origin.y, Origin.z + Distance);
		}

		Origin.x = x2;

		Distance = (dot(Triangle->N, Origin) + Triangle->D) / NdotR;

		if(Distance >= 0.0f && Distance <= Size)
		{
			return Triangle->Inside(Origin.x, Origin.y, Origin.z + Distance);
		}

		Origin.y = y2;

		Distance = (dot(Triangle->N, Origin) + Triangle->D) / NdotR;

		if(Distance >= 0.0f && Distance <= Size)
		{
			return Triangle->Inside(Origin.x, Origin.y, Origin.z + Distance);
		}

		Origin.x = x1;

		Distance = (dot(Triangle->N, Origin) + Triangle->D) / NdotR;

		if(Distance >= 0.0f && Distance <= Size)
		{
			return Triangle->Inside(Origin.x, Origin.y, Origin.z + Distance);
		}
	}

	return false;
}

bool CVoxel::IntersectFacesX(CTriangle *Triangle, float D1, float D2)
{
	vec3 *Origin = (vec3*)&Triangle->a;
	vec3 *Ray = (vec3*)&Triangle->ab;
	float *Length = &Triangle->lab;

	float NdotR, d, y, z;

	for(int i = 0; i < 3; i++)
	{
		NdotR = -Ray->x;

		if(NdotR != 0.0f)
		{
			d = (Origin->x - D1) / NdotR;

			if(0.0f <= d && d <= *Length)
			{
				y = Ray->y * d + Origin->y;

				if(Min.y <= y && y <= Max.y)
				{
					z = Ray->z * d + Origin->z;

					if(Min.z <= z && z <= Max.z)
					{
						return true;
					}
				}
			}

			d = (Origin->x - D2) / NdotR;

			if(0.0f <= d && d <= *Length)
			{
				y = Ray->y * d + Origin->y;

				if(Min.y <= y && y <= Max.y)
				{
					z = Ray->z * d + Origin->z;

					if(Min.z <= z && z <= Max.z)
					{
						return true;
					}
				}
			}
		}

		Ray++;
		Origin++;
		Length++;
	}

	return false;
}

bool CVoxel::IntersectFacesY(CTriangle *Triangle, float D1, float D2)
{
	vec3 *Origin = (vec3*)&Triangle->a;
	vec3 *Ray = (vec3*)&Triangle->ab;
	float *Length = &Triangle->lab;

	float NdotR, d, x, z;

	for(int i = 0; i < 3; i++)
	{
		NdotR = -Ray->y;

		if(NdotR != 0.0f)
		{
			d = (Origin->y - D1) / NdotR;

			if(0.0f <= d && d <= *Length)
			{
				x = Ray->x * d + Origin->x;

				if(Min.x <= x && x <= Max.x)
				{
					z = Ray->z * d + Origin->z;

					if(Min.z <= z && z <= Max.z)
					{
						return true;
					}
				}
			}

			d = (Origin->y - D2) / NdotR;

			if(0.0f <= d && d <= *Length)
			{
				x = Ray->x * d + Origin->x;

				if(Min.x <= x && x <= Max.x)
				{
					z = Ray->z * d + Origin->z;

					if(Min.z <= z && z <= Max.z)
					{
						return true;
					}
				}
			}
		}

		Ray++;
		Origin++;
		Length++;
	}

	return false;
}

bool CVoxel::IntersectFacesZ(CTriangle *Triangle, float D1, float D2)
{
	vec3 *Origin = (vec3*)&Triangle->a;
	vec3 *Ray = (vec3*)&Triangle->ab;
	float *Length = &Triangle->lab;

	float NdotR, d, x, y;

	for(int i = 0; i < 3; i++)
	{
		NdotR = -Ray->z;

		if(NdotR != 0.0f)
		{
			d = (Origin->z - D1) / NdotR;

			if(0.0f <= d && d <= *Length)
			{
				x = Ray->x * d + Origin->x;

				if(Min.x <= x && x <= Max.x)
				{
					y = Ray->y * d + Origin->y;

					if(Min.y <= y && y <= Max.y)
					{
						return true;
					}
				}
			}

			d = (Origin->z - D2) / NdotR;

			if(0.0f <= d && d <= *Length)
			{
				x = Ray->x * d + Origin->x;

				if(Min.x <= x && x <= Max.x)
				{
					y = Ray->y * d + Origin->y;

					if(Min.y <= y && y <= Max.y)
					{
						return true;
					}
				}
			}
		}

		Ray++;
		Origin++;
		Length++;
	}

	return false;
}

bool CVoxel::Intersect(CTriangle *Triangle)
{
	if(Inside(Triangle->a)) return true;
	if(Inside(Triangle->b)) return true;
	if(Inside(Triangle->c)) return true;

	if(IntersectFacesX(Triangle, Min.x, Max.x)) return true;
	if(IntersectFacesY(Triangle, Min.y, Max.y)) return true;
	if(IntersectFacesZ(Triangle, Min.z, Max.z)) return true;

	if(IntersectEdgesX(Triangle, Min.x, Min.y, Max.y, Min.z, Max.z)) return true;
	if(IntersectEdgesY(Triangle, Min.y, Min.x, Max.x, Min.z, Max.z)) return true;
	if(IntersectEdgesZ(Triangle, Min.z, Min.x, Max.x, Min.y, Max.y)) return true;

	return false;
}

void CVoxel::Render()
{
	glBegin(GL_LINES);

	glVertex3f(Min.x, Min.y, Min.z); glVertex3f(Max.x, Min.y, Min.z);
	glVertex3f(Max.x, Min.y, Min.z); glVertex3f(Max.x, Min.y, Max.z);
	glVertex3f(Max.x, Min.y, Max.z); glVertex3f(Min.x, Min.y, Max.z);
	glVertex3f(Min.x, Min.y, Max.z); glVertex3f(Min.x, Min.y, Min.z);

	glVertex3f(Min.x, Min.y, Min.z); glVertex3f(Min.x, Max.y, Min.z);
	glVertex3f(Max.x, Min.y, Min.z); glVertex3f(Max.x, Max.y, Min.z);
	glVertex3f(Max.x, Min.y, Max.z); glVertex3f(Max.x, Max.y, Max.z);
	glVertex3f(Min.x, Min.y, Max.z); glVertex3f(Min.x, Max.y, Max.z);

	glVertex3f(Min.x, Max.y, Min.z); glVertex3f(Max.x, Max.y, Min.z);
	glVertex3f(Max.x, Max.y, Min.z); glVertex3f(Max.x, Max.y, Max.z);
	glVertex3f(Max.x, Max.y, Max.z); glVertex3f(Min.x, Max.y, Max.z);
	glVertex3f(Min.x, Max.y, Max.z); glVertex3f(Min.x, Max.y, Min.z);

	glEnd();
}

void CVoxel::Set(const vec3 &Min, float Size)
{
	this->Size = Size;
	this->Min = Min;
	this->Max = this->Min + Size;
	this->MinE = this->Min - 0.001f;
	this->MaxE = this->Max + 0.001f;
}

// ----------------------------------------------------------------------------------------------------------------------------

CUniformGrid::CUniformGrid()
{
	X = Y = Z = Xm1 = Ym1 = Zm1 = XY = XYZ = 0;
	VoxelSize = 0.0f;
	Voxels = NULL;
}

CUniformGrid::~CUniformGrid()
{
}

void CUniformGrid::Delete()
{
	if(Voxels != NULL)
	{
		for(int i = 0; i < XYZ; i++)
		{
			Voxels[i].Delete();
		}

		Min = Max = vec3(0.0f);
		X = Y = Z = Xm1 = Ym1 = Zm1 = XY = XYZ = 0;
		VoxelSize = 0.0f;
		delete [] Voxels;
		Voxels = NULL;
	}
}

void CUniformGrid::Generate(CTriangle *Triangles, int TrianglesCount, float VoxelSize)
{
	Delete();

	if(Triangles != NULL && TrianglesCount > 0)
	{
		this->VoxelSize = VoxelSize;

		CTriangle *LastTriangle = Triangles + TrianglesCount;

		Min = Max = Triangles->a;

		for(CTriangle *Triangle = Triangles; Triangle < LastTriangle; Triangle++)
		{
			if(Triangle->a.x < Min.x) Min.x = Triangle->a.x;
			if(Triangle->a.y < Min.y) Min.y = Triangle->a.y;
			if(Triangle->a.z < Min.z) Min.z = Triangle->a.z;

			if(Triangle->b.x < Min.x) Min.x = Triangle->b.x;
			if(Triangle->b.y < Min.y) Min.y = Triangle->b.y;
			if(Triangle->b.z < Min.z) Min.z = Triangle->b.z;

			if(Triangle->c.x < Min.x) Min.x = Triangle->c.x;
			if(Triangle->c.y < Min.y) Min.y = Triangle->c.y;
			if(Triangle->c.z < Min.z) Min.z = Triangle->c.z;

			if(Triangle->a.x > Max.x) Max.x = Triangle->a.x;
			if(Triangle->a.y > Max.y) Max.y = Triangle->a.y;
			if(Triangle->a.z > Max.z) Max.z = Triangle->a.z;

			if(Triangle->b.x > Max.x) Max.x = Triangle->b.x;
			if(Triangle->b.y > Max.y) Max.y = Triangle->b.y;
			if(Triangle->b.z > Max.z) Max.z = Triangle->b.z;

			if(Triangle->c.x > Max.x) Max.x = Triangle->c.x;
			if(Triangle->c.y > Max.y) Max.y = Triangle->c.y;
			if(Triangle->c.z > Max.z) Max.z = Triangle->c.z;
		}

		Min /= VoxelSize; Max /= VoxelSize;

		Min.x = floor(Min.x); Max.x = ceil(Max.x);
		Min.y = floor(Min.y); Max.y = ceil(Max.y);
		Min.z = floor(Min.z); Max.z = ceil(Max.z);

		if(Min.x == Max.x) Max.x += 1.0f;
		if(Min.y == Max.y) Max.y += 1.0f;
		if(Min.z == Max.z) Max.z += 1.0f;

		X = (int)(Max.x - Min.x); Xm1 = X - 1;
		Y = (int)(Max.y - Min.y); Ym1 = Y - 1;
		Z = (int)(Max.z - Min.z); Zm1 = Z - 1;

		XY = X * Y;
		XYZ = XY * Z;

		Min *= VoxelSize; Max *= VoxelSize;

		Voxels = new CVoxel[XYZ];

		for(int z = 0; z < Z; z++)
		{
			for(int y = 0; y < Y; y++)
			{
				for(int x = 0; x < X; x++)
				{
					Voxels[XY * z + X * y + x].Set(VoxelToWorld(vec3((float)x, (float)y, (float)z)), VoxelSize);
				}
			}
		}

		for(CTriangle *Triangle = Triangles; Triangle < LastTriangle; Triangle++)
		{
			vec3 tmin = Triangle->a, tmax = tmin;

			if(Triangle->b.x < tmin.x) tmin.x = Triangle->b.x;
			if(Triangle->b.y < tmin.y) tmin.y = Triangle->b.y;
			if(Triangle->b.z < tmin.z) tmin.z = Triangle->b.z;

			if(Triangle->b.x > tmax.x) tmax.x = Triangle->b.x;
			if(Triangle->b.y > tmax.y) tmax.y = Triangle->b.y;
			if(Triangle->b.z > tmax.z) tmax.z = Triangle->b.z;

			if(Triangle->c.x < tmin.x) tmin.x = Triangle->c.x;
			if(Triangle->c.y < tmin.y) tmin.y = Triangle->c.y;
			if(Triangle->c.z < tmin.z) tmin.z = Triangle->c.z;

			if(Triangle->c.x > tmax.x) tmax.x = Triangle->c.x;
			if(Triangle->c.y > tmax.y) tmax.y = Triangle->c.y;
			if(Triangle->c.z > tmax.z) tmax.z = Triangle->c.z;

			int vminx = WorldToVoxelX(tmin.x), vmaxx = WorldToVoxelX(tmax.x);
			int vminy = WorldToVoxelY(tmin.y), vmaxy = WorldToVoxelY(tmax.y);
			int vminz = WorldToVoxelZ(tmin.z), vmaxz = WorldToVoxelZ(tmax.z);

			if(vminx >= X) vminx = Xm1; if(vmaxx >= X) vmaxx = Xm1;
			if(vminy >= Y) vminy = Ym1; if(vmaxy >= Y) vmaxy = Ym1;
			if(vminz >= Z) vminz = Zm1; if(vmaxz >= Z) vmaxz = Zm1;

			for(int z = vminz; z <= vmaxz; z++)
			{
				for(int y = vminy; y <= vmaxy; y++)
				{
					for(int x = vminx; x <= vmaxx; x++)
					{
						CVoxel *Voxel = Voxels + (XY * z + X * y + x);

						if(Voxel->Intersect(Triangle))
						{
							Voxel->Add(Triangle);
						}
					}
				}
			}
		}
	}
}

void CUniformGrid::RenderGrid()
{
	for(int z = 0; z < Z; z++)
	{
		for(int y = 0; y < Y; y++)
		{
			for(int x = 0; x < X; x++)
			{
				Voxels[XY * z + X * y + x].Render();
			}
		}
	}
}

vec3 CUniformGrid::Traverse(const vec3 &Voxel, const vec3 &Origin, const vec3 &Ray)
{
	vec3 voxel = Voxel, step, out, t, delta = VoxelSize / Ray;

	if(Ray.x < 0.0f)
	{
		step.x = -1.0f;
		out.x = voxel.x <= 0.0f ? voxel.x - 1.0f : -1.0f;
		t.x = (VoxelToWorldX(voxel.x) - Origin.x) / Ray.x;
	}
	else
	{
		step.x = 1.0f;
		out.x = voxel.x >= X ? voxel.x + 1 : X;
		t.x = (VoxelToWorldX(voxel.x + 1.0f) - Origin.x) / Ray.x;
	}

	if(Ray.y < 0.0f)
	{
		step.y = -1.0f;
		out.y = voxel.y <= 0.0f ? voxel.y - 1.0f : -1.0f;
		t.y = (VoxelToWorldY(voxel.y) - Origin.y) / Ray.y;
	}
	else
	{
		step.y = 1.0f;
		out.y = voxel.y >= Y ? voxel.y + 1 : Y;
		t.y = (VoxelToWorldY(voxel.y + 1.0f) - Origin.y) / Ray.y;
	}

	if(Ray.z < 0.0f)
	{
		step.z = -1.0f;
		out.z = voxel.z <= 0.0f ? voxel.z - 1.0f : -1.0f;
		t.z = (VoxelToWorldZ(voxel.z) - Origin.z) / Ray.z;
	}
	else
	{
		step.z = 1.0f;
		out.z = voxel.z >= Z ? voxel.z + 1 : Z;
		t.z = (VoxelToWorldZ(voxel.z + 1.0f) - Origin.z) / Ray.z;
	}

	delta *= step;

	while(1)
	{
		int x = (int)voxel.x, y = (int)voxel.y, z = (int)voxel.z;

		if(x >= 0 && x < X && y >= 0 && y < Y && z >= 0 && z < Z)
		{
			CVoxel *Voxel = Voxels + (XY * z + X * y + x);

			glLineWidth(1.0f);
			glColor3f(1.0f, 1.0f, 1.0f);
			Voxel->Render();

			CTriangle **Triangles = Voxel->Triangles, *Triangle;
			int TrianglesCount = Voxel->TrianglesCount;

			RTData rtdata;

			for(int i = 0; i < TrianglesCount; i++)
			{
				Triangle = Triangles[i];

				if(Triangle->Intersect(Origin, Ray, rtdata.Distance, rtdata.TestDistance, rtdata.TestPoint))
				{
					if(Voxel->Inside(rtdata.TestPoint))
					{
						rtdata.Point = rtdata.TestPoint;
						rtdata.Distance = rtdata.TestDistance;
						rtdata.Triangle = Triangle;
					}
				}
			}

			for(int i = 0; i < TrianglesCount; i++)
			{
				Triangle = Triangles[i];

				if(Triangle == rtdata.Triangle)
				{
					glColor3f(1.0f, 1.0f, 1.0f);
					glPolygonOffset(-1.0f, -1.0f);
					glEnable(GL_POLYGON_OFFSET_FILL);
					glEnable(GL_POLYGON_OFFSET_LINE);
				}
				else
				{
					glColor3fv(&Triangle->Color);
				}

				glBegin(GL_TRIANGLES);
				glVertex3fv(&Triangle->a);
				glVertex3fv(&Triangle->b);
				glVertex3fv(&Triangle->c);
				glEnd();

				glLineWidth(2.0f);
				glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);

				if(Triangle == rtdata.Triangle)
				{
					glColor3f(0.0f, 1.0f, 0.0f);
				}
				else
				{
					glColor3f(1.0f, 1.0f, 1.0f);
				}

				glBegin(GL_TRIANGLES);
				glVertex3fv(&Triangle->a);
				glVertex3fv(&Triangle->b);
				glVertex3fv(&Triangle->c);
				glEnd();

				glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
				
				if(Triangle == rtdata.Triangle)
				{
					glDisable(GL_POLYGON_OFFSET_LINE);
					glDisable(GL_POLYGON_OFFSET_FILL);
					glPolygonOffset(0.0f, 0.0f);
				}
			}

			if(rtdata.Triangle)
			{
				rtdata.Color = rtdata.Triangle->Color;

				float NdotL = -dot(rtdata.Triangle->N, Ray);

				if(NdotL < 0.0f)
				{
					NdotL = 0.0f;
				}

				rtdata.Color *= 0.75f * NdotL + 0.25f;

				return rtdata.Color;
			}
		}

		float min = t.x;

		if(t.y < min) min = t.y;
		if(t.z < min) min = t.z;

		if(t.x == min)
		{
		   voxel.x += step.x;
		   if(voxel.x == out.x) break;
		   t.x += delta.x;
		}

		if(t.y == min)
		{
		   voxel.y += step.y;
		   if(voxel.y == out.y) break;
		   t.y += delta.y;
		}

		if(t.z == min)
		{
		   voxel.z += step.z;
		   if(voxel.z == out.z) break;
		   t.z += delta.z;
		}
	}

	return vec3(0.0f);
}

float CUniformGrid::VoxelToWorldX(float x)
{
	return x * VoxelSize + Min.x;
}

float CUniformGrid::VoxelToWorldY(float y)
{
	return y * VoxelSize + Min.y;
}

float CUniformGrid::VoxelToWorldZ(float z)
{
	return z * VoxelSize + Min.z;
}

vec3 CUniformGrid::VoxelToWorld(const vec3 &Voxel)
{
	return Voxel * VoxelSize + Min;
}

int CUniformGrid::WorldToVoxelX(float x)
{
	return (int)floor((x - Min.x) / VoxelSize);
}

int CUniformGrid::WorldToVoxelY(float y)
{
	return (int)floor((y - Min.y) / VoxelSize);
}

int CUniformGrid::WorldToVoxelZ(float z)
{
	return (int)floor((z - Min.z) / VoxelSize);
}

vec3 CUniformGrid::WorldToVoxel(const vec3 &World)
{
	vec3 voxel = (World - Min) / VoxelSize;

	voxel.x = floor(voxel.x);
	voxel.y = floor(voxel.y);
	voxel.z = floor(voxel.z);

	return voxel;
}

// ----------------------------------------------------------------------------------------------------------------------------

COpenGLRenderer::COpenGLRenderer()
{
	Camera.SetViewMatrixPointer(&ViewMatrix, &ViewMatrixInverse);
}

COpenGLRenderer::~COpenGLRenderer()
{
}

bool COpenGLRenderer::Init()
{
	srand(GetTickCount());

	int BoxesCount = 128;

	TrianglesCount = 12 * BoxesCount;

	Triangles = new CTriangle[TrianglesCount];

	int t = 0;

	for(int i = 0; i < BoxesCount; i++)
	{
		vec3 m = -4.0f + 8.0f * vec3((float)rand() / (float)RAND_MAX, (float)rand() / (float)RAND_MAX, (float)rand() / (float)RAND_MAX);
		vec3 s = 0.25f + 1.5f * vec3((float)rand() / (float)RAND_MAX, (float)rand() / (float)RAND_MAX, (float)rand() / (float)RAND_MAX);
		vec3 color = vec3((float)rand() / (float)RAND_MAX, (float)rand() / (float)RAND_MAX, (float)rand() / (float)RAND_MAX);

		mat3x3 RS;
		
		RS = RS * mat3x3(rotate(360.0f * (float)rand() / (float)RAND_MAX, vec3(1.0f, 0.0f, 0.0f)));
		RS = RS * mat3x3(rotate(360.0f * (float)rand() / (float)RAND_MAX, vec3(0.0f, 1.0f, 0.0f)));
		RS = RS * mat3x3(rotate(360.0f * (float)rand() / (float)RAND_MAX, vec3(0.0f, 0.0f, 1.0f)));
		
		RS = RS * mat3x3(scale(s.x, s.y, s.z));

		Triangles[t++] = CTriangle(m + RS * vec3( 0.5f, -0.5f,  0.5f), m + RS * vec3( 0.5f, -0.5f, -0.5f), m + RS * vec3( 0.5f,  0.5f, -0.5f), color);
		Triangles[t++] = CTriangle(m + RS * vec3( 0.5f,  0.5f, -0.5f), m + RS * vec3( 0.5f,  0.5f,  0.5f), m + RS * vec3( 0.5f, -0.5f,  0.5f), color);
		Triangles[t++] = CTriangle(m + RS * vec3(-0.5f, -0.5f, -0.5f), m + RS * vec3(-0.5f, -0.5f,  0.5f), m + RS * vec3(-0.5f,  0.5f,  0.5f), color);
		Triangles[t++] = CTriangle(m + RS * vec3(-0.5f,  0.5f,  0.5f), m + RS * vec3(-0.5f,  0.5f, -0.5f), m + RS * vec3(-0.5f, -0.5f, -0.5f), color);
		Triangles[t++] = CTriangle(m + RS * vec3(-0.5f,  0.5f,  0.5f), m + RS * vec3( 0.5f,  0.5f,  0.5f), m + RS * vec3( 0.5f,  0.5f, -0.5f), color);
		Triangles[t++] = CTriangle(m + RS * vec3( 0.5f,  0.5f, -0.5f), m + RS * vec3(-0.5f,  0.5f, -0.5f), m + RS * vec3(-0.5f,  0.5f,  0.5f), color);
		Triangles[t++] = CTriangle(m + RS * vec3(-0.5f, -0.5f, -0.5f), m + RS * vec3( 0.5f, -0.5f, -0.5f), m + RS * vec3( 0.5f, -0.5f,  0.5f), color);
		Triangles[t++] = CTriangle(m + RS * vec3( 0.5f, -0.5f,  0.5f), m + RS * vec3(-0.5f, -0.5f,  0.5f), m + RS * vec3(-0.5f, -0.5f, -0.5f), color);
		Triangles[t++] = CTriangle(m + RS * vec3(-0.5f, -0.5f,  0.5f), m + RS * vec3( 0.5f, -0.5f,  0.5f), m + RS * vec3( 0.5f,  0.5f,  0.5f), color);
		Triangles[t++] = CTriangle(m + RS * vec3( 0.5f,  0.5f,  0.5f), m + RS * vec3(-0.5f,  0.5f,  0.5f), m + RS * vec3(-0.5f, -0.5f,  0.5f), color);
		Triangles[t++] = CTriangle(m + RS * vec3( 0.5f, -0.5f, -0.5f), m + RS * vec3(-0.5f, -0.5f, -0.5f), m + RS * vec3(-0.5f,  0.5f, -0.5f), color);
		Triangles[t++] = CTriangle(m + RS * vec3(-0.5f,  0.5f, -0.5f), m + RS * vec3( 0.5f,  0.5f, -0.5f), m + RS * vec3( 0.5f, -0.5f, -0.5f), color);
	}

	UniformGrid.Generate(Triangles, TrianglesCount);

	Camera.Look(vec3(0.0f, 0.0f, 10.0f), vec3(0.0f, 0.0f, 0.0f), true);

	return true;
}

void COpenGLRenderer::Render(float FrameTime)
{
	static float a = 0.0f;

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glEnable(GL_DEPTH_TEST);

	glMatrixMode(GL_MODELVIEW);
	glLoadMatrixf(&ViewMatrix);

	if(Start.x != End.x && Start.y != End.y && Start.z != End.z)
	{
		glLineWidth(2.0f);
		glColor3f(1.0f, 0.0f, 0.0f);
		glBegin(GL_LINES);
		glVertex3fv(&Start);
		glVertex3fv(&End);
		glEnd();

		UniformGrid.Traverse(UniformGrid.WorldToVoxel(Start), Start, normalize(End - Start));
	}
	else
	{
		glLineWidth(1.0f);
		glColor3f(1.0f, 1.0f, 1.0f);
		UniformGrid.RenderGrid();

		glBegin(GL_TRIANGLES);

		for(int i = 0; i < TrianglesCount; i++)
		{
			glColor3fv(&Triangles[i].Color);
			glVertex3fv(&Triangles[i].a);
			glVertex3fv(&Triangles[i].b);
			glVertex3fv(&Triangles[i].c);
		}

		glEnd();
	}

	glDisable(GL_DEPTH_TEST);
}

void COpenGLRenderer::Resize(int Width, int Height)
{
	this->Width = Width;
	this->Height = Height;

	glViewport(0, 0, Width, Height);

	ProjectionMatrix = perspective(45.0f, (float)Width / (float)Height, 0.125f, 512.0f);
	ProjectionMatrixInverse = inverse(ProjectionMatrix);

	glMatrixMode(GL_PROJECTION);
	glLoadMatrixf(&ProjectionMatrix);
}

void COpenGLRenderer::Destroy()
{
	UniformGrid.Delete();

	if(Triangles != NULL)
	{
		delete [] Triangles;
		Triangles = NULL;
		TrianglesCount = 0;
	}
}

void COpenGLRenderer::Check(int x, int y)
{
	vec4 v = ViewMatrixInverse * ProjectionMatrixInverse * BiasMatrixInverse * vec4((float)x / (float)Width, (float)(Height - 1 - y) / (float)Height, 1.0f, 1.0f);
	v /= v.w;

	Start = Camera.Position;
	End = *(vec3*)&v;
}

// ----------------------------------------------------------------------------------------------------------------------------

COpenGLRenderer OpenGLRenderer;

// ----------------------------------------------------------------------------------------------------------------------------

CString ModuleDirectory, ErrorLog;

// ----------------------------------------------------------------------------------------------------------------------------

void GetModuleDirectory()
{
	char *moduledirectory = new char[256];
	GetModuleFileName(GetModuleHandle(NULL), moduledirectory, 256);
	*(strrchr(moduledirectory, '\\') + 1) = 0;
	ModuleDirectory = moduledirectory;
	delete [] moduledirectory;
}

// ----------------------------------------------------------------------------------------------------------------------------

COpenGLView::COpenGLView()
{
}

COpenGLView::~COpenGLView()
{
}

bool COpenGLView::Init(HINSTANCE hInstance, char *Title, int Width, int Height, int Samples)
{
	this->Title = Title;
	this->Width = Width;
	this->Height = Height;

	WNDCLASSEX WndClassEx;

	memset(&WndClassEx, 0, sizeof(WNDCLASSEX));

	WndClassEx.cbSize = sizeof(WNDCLASSEX);
	WndClassEx.style = CS_OWNDC | CS_HREDRAW | CS_VREDRAW;
	WndClassEx.lpfnWndProc = WndProc;
	WndClassEx.hInstance = hInstance;
	WndClassEx.hIcon = LoadIcon(NULL, IDI_APPLICATION);
	WndClassEx.hIconSm = LoadIcon(NULL, IDI_APPLICATION);
	WndClassEx.hCursor = LoadCursor(NULL, IDC_ARROW);
	WndClassEx.lpszClassName = "Win32OpenGLWindowClass";

	if(RegisterClassEx(&WndClassEx) == 0)
	{
		ErrorLog.Set("RegisterClassEx failed!");
		return false;
	}

	DWORD Style = WS_OVERLAPPEDWINDOW | WS_CLIPSIBLINGS | WS_CLIPCHILDREN;

	hWnd = CreateWindowEx(WS_EX_APPWINDOW, WndClassEx.lpszClassName, Title, Style, 0, 0, Width, Height, NULL, NULL, hInstance, NULL);

	if(hWnd == NULL)
	{
		ErrorLog.Set("CreateWindowEx failed!");
		return false;
	}

	HDC hDC = GetDC(hWnd);

	if(hDC == NULL)
	{
		ErrorLog.Set("GetDC failed!");
		return false;
	}

	PIXELFORMATDESCRIPTOR pfd;

	memset(&pfd, 0, sizeof(PIXELFORMATDESCRIPTOR));

	pfd.nSize = sizeof(PIXELFORMATDESCRIPTOR);
	pfd.nVersion = 1;
	pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
	pfd.iPixelType = PFD_TYPE_RGBA;
	pfd.cColorBits = 32;
	pfd.cDepthBits = 24;
	pfd.iLayerType = PFD_MAIN_PLANE;

	int PixelFormat = ChoosePixelFormat(hDC, &pfd);

	if(PixelFormat == 0)
	{
		ErrorLog.Set("ChoosePixelFormat failed!");
		return false;
	}

	static int MSAAPixelFormat = 0;

	if(SetPixelFormat(hDC, MSAAPixelFormat == 0 ? PixelFormat : MSAAPixelFormat, &pfd) == FALSE)
	{
		ErrorLog.Set("SetPixelFormat failed!");
		return false;
	}

	hGLRC = wglCreateContext(hDC);

	if(hGLRC == NULL)
	{
		ErrorLog.Set("wglCreateContext failed!");
		return false;
	}

	if(wglMakeCurrent(hDC, hGLRC) == FALSE)
	{
		ErrorLog.Set("wglMakeCurrent failed!");
		return false;
	}

	if(glewInit() != GLEW_OK)
	{
		ErrorLog.Set("glewInit failed!");
		return false;
	}

	if(!GLEW_VERSION_2_1)
	{
		ErrorLog.Set("OpenGL 2.1 not supported!");
		return false;
	}

	if(MSAAPixelFormat == 0 && Samples > 0)
	{
		if(GLEW_ARB_multisample && WGLEW_ARB_pixel_format)
		{
			while(Samples > 0)
			{
				UINT NumFormats = 0;

				int PFAttribs[] =
				{
					WGL_DRAW_TO_WINDOW_ARB, GL_TRUE,
					WGL_SUPPORT_OPENGL_ARB, GL_TRUE,
					WGL_DOUBLE_BUFFER_ARB, GL_TRUE,
					WGL_PIXEL_TYPE_ARB, WGL_TYPE_RGBA_ARB,
					WGL_COLOR_BITS_ARB, 32,
					WGL_DEPTH_BITS_ARB, 24,
					WGL_ACCELERATION_ARB, WGL_FULL_ACCELERATION_ARB,
					WGL_SAMPLE_BUFFERS_ARB, GL_TRUE,
					WGL_SAMPLES_ARB, Samples,
					0
				};

				if(wglChoosePixelFormatARB(hDC, PFAttribs, NULL, 1, &MSAAPixelFormat, &NumFormats) == TRUE && NumFormats > 0) break;

				Samples--;
			}

			wglDeleteContext(hGLRC);
			DestroyWindow(hWnd);
			UnregisterClass(WndClassEx.lpszClassName, hInstance);

			return Init(hInstance, Title, Width, Height, Samples);
		}
		else
		{
			Samples = 0;
		}
	}

	this->Samples = Samples;

	GetModuleDirectory();

	glGetIntegerv(GL_MAX_TEXTURE_SIZE, &gl_max_texture_size);

	if(GLEW_EXT_texture_filter_anisotropic)
	{
		glGetIntegerv(GL_MAX_TEXTURE_MAX_ANISOTROPY_EXT, &gl_max_texture_max_anisotropy_ext);
	}

	if(WGLEW_EXT_swap_control)
	{
		wglSwapIntervalEXT(0);
	}

	return OpenGLRenderer.Init();
}

void COpenGLView::Show(bool Maximized)
{
	RECT dRect, wRect, cRect;

	GetWindowRect(GetDesktopWindow(), &dRect);
	GetWindowRect(hWnd, &wRect);
	GetClientRect(hWnd, &cRect);

	wRect.right += Width - cRect.right;
	wRect.bottom += Height - cRect.bottom;
	wRect.right -= wRect.left;
	wRect.bottom -= wRect.top;
	wRect.left = dRect.right / 2 - wRect.right / 2;
	wRect.top = dRect.bottom / 2 - wRect.bottom / 2;

	MoveWindow(hWnd, wRect.left, wRect.top, wRect.right, wRect.bottom, FALSE);

	ShowWindow(hWnd, Maximized ? SW_SHOWMAXIMIZED : SW_SHOWNORMAL);
}

void COpenGLView::MessageLoop()
{
	MSG Msg;

	while(GetMessage(&Msg, NULL, 0, 0) > 0)
	{
		TranslateMessage(&Msg);
		DispatchMessage(&Msg);
	}
}

void COpenGLView::Destroy()
{
	if(GLEW_VERSION_2_1)
	{
		OpenGLRenderer.Destroy();
	}

	wglDeleteContext(hGLRC);
	DestroyWindow(hWnd);
}

void COpenGLView::OnKeyDown(UINT Key)
{
	float increment = 0.1f;

	switch(Key)
	{
		case VK_ESCAPE:
			OpenGLRenderer.Start = OpenGLRenderer.End = vec3(0.0f);
			break;
	}
}

void COpenGLView::OnLButtonDown(int X, int Y)
{
	OpenGLRenderer.Check(X, Y);
}

void COpenGLView::OnMouseMove(int X, int Y)
{
	if(GetKeyState(VK_RBUTTON) & 0x80)
	{
		Camera.OnMouseMove(LastX - X, LastY - Y);

		LastX = X;
		LastY = Y;
	}
}

void COpenGLView::OnMouseWheel(short zDelta)
{
	Camera.OnMouseWheel(zDelta);
}

void COpenGLView::OnPaint()
{
	static DWORD LastFPSTime = GetTickCount(), LastFrameTime = LastFPSTime, FPS = 0;

	PAINTSTRUCT ps;

	HDC hDC = BeginPaint(hWnd, &ps);

	DWORD Time = GetTickCount();

	float FrameTime = (Time - LastFrameTime) * 0.001f;

	LastFrameTime = Time;

	if(Time - LastFPSTime > 1000)
	{
		CString Text = Title;

		if(OpenGLRenderer.Text[0] != 0)
		{
			Text.Append(" - " + OpenGLRenderer.Text);
		}

		Text.Append(" - %dx%d", Width, Height);
		Text.Append(", ATF %dx", gl_max_texture_max_anisotropy_ext);
		Text.Append(", MSAA %dx", Samples);
		Text.Append(", FPS: %d", FPS);
		Text.Append(" - %s", glGetString(GL_RENDERER));
		
		SetWindowText(hWnd, Text);

		LastFPSTime = Time;
		FPS = 0;
	}
	else
	{
		FPS++;
	}

	BYTE Keys = 0x00;

	if(GetKeyState('W') & 0x80) Keys |= 0x01;
	if(GetKeyState('S') & 0x80) Keys |= 0x02;
	if(GetKeyState('A') & 0x80) Keys |= 0x04;
	if(GetKeyState('D') & 0x80) Keys |= 0x08;
	if(GetKeyState('R') & 0x80) Keys |= 0x10;
	if(GetKeyState('F') & 0x80) Keys |= 0x20;

	if(GetKeyState(VK_SHIFT) & 0x80) Keys |= 0x40;
	if(GetKeyState(VK_CONTROL) & 0x80) Keys |= 0x80;

	if(Keys & 0x3F)
	{
		Camera.Move(Camera.OnKeys(Keys, FrameTime));
	}

	OpenGLRenderer.Render(FrameTime);

	SwapBuffers(hDC);

	EndPaint(hWnd, &ps);

	InvalidateRect(hWnd, NULL, FALSE);
}

void COpenGLView::OnRButtonDown(int X, int Y)
{
	LastX = X;
	LastY = Y;
}

void COpenGLView::OnSize(int Width, int Height)
{
	this->Width = Width;
	this->Height = Height;

	OpenGLRenderer.Resize(Width, Height);
}

// ----------------------------------------------------------------------------------------------------------------------------

COpenGLView OpenGLView;

// ----------------------------------------------------------------------------------------------------------------------------

LRESULT CALLBACK WndProc(HWND hWnd, UINT uiMsg, WPARAM wParam, LPARAM lParam)
{
	switch(uiMsg)
	{
		case WM_CLOSE:
			PostQuitMessage(0);
			break;

		case WM_MOUSEMOVE:
			OpenGLView.OnMouseMove(LOWORD(lParam), HIWORD(lParam));
			break;

		case 0x020A: // WM_MOUSWHEEL
			OpenGLView.OnMouseWheel(HIWORD(wParam));
			break;

		case WM_KEYDOWN:
			OpenGLView.OnKeyDown((UINT)wParam);
			break;

		case WM_LBUTTONDOWN:
			OpenGLView.OnLButtonDown(LOWORD(lParam), HIWORD(lParam));
			break;

		case WM_PAINT:
			OpenGLView.OnPaint();
			break;

		case WM_RBUTTONDOWN:
			OpenGLView.OnRButtonDown(LOWORD(lParam), HIWORD(lParam));
			break;

		case WM_SIZE:
			OpenGLView.OnSize(LOWORD(lParam), HIWORD(lParam));
			break;

		default:
			return DefWindowProc(hWnd, uiMsg, wParam, lParam);
	}

	return 0;
}

// ----------------------------------------------------------------------------------------------------------------------------

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR sCmdLine, int iShow)
{
	char *AppName = "Traversing uniform grid visualization";

	if(OpenGLView.Init(hInstance, AppName, 800, 600, 4))
	{
		OpenGLView.Show();
		OpenGLView.MessageLoop();
	}
	else
	{
		MessageBox(NULL, ErrorLog, AppName, MB_OK | MB_ICONERROR);
	}

	OpenGLView.Destroy();

	return 0;
}
