// ----------------------------------------------------------------------------------------------------------------------------
//
// Version 2.02
//
// ----------------------------------------------------------------------------------------------------------------------------

#include <windows.h>

#include "glmath.h"
#include "string.h"

#include <gl/glew.h> // http://glew.sourceforge.net/
#include <gl/wglew.h>

#include <FreeImage.h> // http://freeimage.sourceforge.net/

// ----------------------------------------------------------------------------------------------------------------------------

#pragma comment(lib, "opengl32.lib")
#pragma comment(lib, "glu32.lib")
#pragma comment(lib, "glew32.lib")
#pragma comment(lib, "FreeImage.lib")

// ----------------------------------------------------------------------------------------------------------------------------

extern CString ModuleDirectory, ErrorLog;

// ----------------------------------------------------------------------------------------------------------------------------

#define BUFFER_SIZE_INCREMENT 1048576

// ----------------------------------------------------------------------------------------------------------------------------

class CBuffer
{
private:
	BYTE *Buffer;
	int BufferSize, Position;

public:
	CBuffer();
	~CBuffer();

	void AddData(void *Data, int DataSize);
	void Empty();
	void *GetData();
	int GetDataSize();

private:
	void SetDefaults();
};

// ----------------------------------------------------------------------------------------------------------------------------

extern int gl_max_texture_size, gl_max_texture_max_anisotropy_ext;

// ----------------------------------------------------------------------------------------------------------------------------

class CTexture
{
protected:
	GLuint Texture;

public:
	int Width, Height;

public:
	CTexture();
	~CTexture();

	operator GLuint ();

	bool LoadTexture2D(char *FileName);
	bool LoadTextureCubeMap(char **FileNames);
	void Destroy();

protected:
	FIBITMAP *CTexture::GetBitmap(char *FileName, int &Width, int &Height, int &BPP);
};

// ----------------------------------------------------------------------------------------------------------------------------

class CShaderProgram
{
protected:
	GLuint VertexShader, FragmentShader, Program;

public:
	GLuint *UniformLocations, *AttribLocations;

public:
	CShaderProgram();
	~CShaderProgram();

	operator GLuint ();

	bool Load(char *VertexShaderFileName, char *FragmentShaderFileName);
	void Destroy();

protected:
	GLuint LoadShader(char *FileName, GLenum Type);
	void SetDefaults();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CFrustum
{
protected:
	vec4 V[8];
	vec3 N[6];
	float D[6];

public:
	CFrustum();
	~CFrustum();

	void Set(mat4x4 &ViewMatrixInverse, mat4x4 &ProjectionMatrixInverse);
	void Set(int AX, int AY, int BX, int BY, int WidthM1, int HeightM1, mat4x4 &ViewMatrixInverse, mat4x4 &ProjectionMatrixInverse);
	void Set(float AX, float AY, float BX, float BY, mat4x4 &ViewMatrixInverse, mat4x4 &ProjectionMatrixInverse);
	bool VertexInside(vec3 &Vertex);
};

// ----------------------------------------------------------------------------------------------------------------------------

class CCamera
{
public:
	vec3 X, Y, Z, Position, Reference;
	mat4x4 ViewMatrix, ViewMatrixInverse, ProjectionMatrix, ProjectionMatrixInverse;
	CFrustum Frustum;

public:
	CCamera();
	~CCamera();

	void Look(const vec3 &Position, const vec3 &Reference, bool RotateAroundReference = false);
	void Move(const vec3 &Movement);
	vec3 OnKeys(BYTE Keys, float FrameTime);
	void OnMouseMove(int dx, int dy);
	void OnMouseWheel(float zDelta);
	void SetPerspectiveProjection(float fovy, float aspect, float n, float f);

private:
	void CalculateViewMatrix();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CSelectedVerticesList
{
protected:
	int *VerticesIndices;
	BYTE *VerticesIndicesFlags;
	int VerticesCount, VerticesIndicesCount;

public:
	CSelectedVerticesList();
	~CSelectedVerticesList();

	void AddVertexIndex(int VertexIndex);
	void Create(int VerticesCount);
	void Destroy();
	void Empty();
	int GetVertexIndex(int Index);
	int GetVerticesIndicesCount();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CTerrain
{
protected:
	int Size, SizeP1;
	float SizeD2, MSizeD2, ODSizeD2, OMODSizeD2, MinHeight, MaxHeight;
	int VerticesCount;
	vec3 *Vertices, *Normals;
	int LinesIndicesCount, QuadsIndicesCount;
	int *LinesIndices, *QuadsIndices;
	GLuint HeightMapTexture;
	CSelectedVerticesList SelectedVerticesIndices;

public:
	CTerrain();
	~CTerrain();

	bool Create(int Size);
	bool Load(char *FileName);
	bool LoadHeightMapTexture(char *FileName, float ScaleHeight = 1.0f);
	bool Save(char *FileName);
	void RenderLines();
	void RenderQuads(bool NormalArray);
	void Destroy();

	GLuint GetHeightMapTexture();
	float GetMinHeight();
	float GetMaxHeight();
	int GetSize();
	float GetSizeD2();
	float GetMSizeD2();
	float GetODSizeD2();
	float GetOMODSizeD2();

	void CalculateMinAndMaxHeights();
	void CalculateNormals();
	void CopyVerticesToHeightMapTexture();
	void Displace(float Displacement);
	float GetHeight(int X, int Z);
	float GetHeight(float X, float Z);
	void GetMinMax(mat4x4 &ViewMatrix, vec3 &Min, vec3 &Max);
	int GetVertexIndex(int X, int Z);
	void Randomize();
	void RenderSelectionBoxAtVertex(int VertexIndex, mat4x4 &ViewMatrix);
	void RenderSelectionBoxesAtSelectedVertices(mat4x4 &ViewMatrix);
	void SelectVertices(CFrustum Frustum);
	void Smooth();
	void UnselectAllVertices();
};

// ----------------------------------------------------------------------------------------------------------------------------

#define SHADOW_MAP_SIZE 4096

// ----------------------------------------------------------------------------------------------------------------------------

class COpenGLRenderer
{
protected:
	int Width, Height, WidthM1, HeightM1;
	mat3x3 NormalMatrix;
	mat4x4 ModelMatrix, OrthoMatrix;

protected:
	CShaderProgram TerrainShader, WaterShader, MapShader;
	CTerrain Terrain;
	GLuint ShadowMap, RotationTexture, FBO;
	int ShadowMapSize;
	float LightAngle;
	vec3 LightPosition, LightDirection;
	mat4x4 LightViewMatrix, LightProjectionMatrix, ShadowMatrix;
	int AX, AY, BX, BY;
	bool RenderLines, RenderWater, DisplayMap, DisplayShadowMap;

public:
	CString Text;

public:
	COpenGLRenderer();
	~COpenGLRenderer();

	bool Init();
	void Render(float FrameTime);
	void Resize(int Width, int Height);
	void Destroy();
	
	void SetText();

	void CalculateShadowMatrix();
	void CheckCameraTerrainPosition();
	void MoveLight(float Angle);
	void RenderShadowMap();

	void OnKeyDown(UINT Key);
	void OnLButtonDown(int X, int Y);
	void OnLButtonUp(int X, int Y);
	void OnMouseMove(int X, int Y);
};

// ----------------------------------------------------------------------------------------------------------------------------

class COpenGLView
{
protected:
	char *Title;
	int Width, Height, Samples;
	HWND hWnd;
	HGLRC hGLRC;

protected:
	int LastX, LastY;

public:
	COpenGLView();
	~COpenGLView();

	bool Init(HINSTANCE hInstance, char *Title, int Width, int Height, int Samples);
	void Show(bool Maximized = false);
	void MessageLoop();
	void Destroy();

	void OnKeyDown(UINT Key);
	void OnLButtonDown(int X, int Y);
	void OnLButtonUp(int X, int Y);
	void OnMouseMove(int X, int Y);
	void OnMouseWheel(short zDelta);
	void OnPaint();
	void OnRButtonDown(int X, int Y);
	void OnSize(int Width, int Height);
};

// ----------------------------------------------------------------------------------------------------------------------------

LRESULT CALLBACK WndProc(HWND hWnd, UINT uiMsg, WPARAM wParam, LPARAM lParam);

// ----------------------------------------------------------------------------------------------------------------------------

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR sCmdLine, int iShow);
