#include "opengl_21_tutorials_win32_framework.h"

// ----------------------------------------------------------------------------------------------------------------------------

CBuffer::CBuffer()
{
	SetDefaults();
}

CBuffer::~CBuffer()
{
	Empty();
}

void CBuffer::AddData(void *Data, int DataSize)
{
	int Remaining = BufferSize - Position;

	if(DataSize > Remaining)
	{
		BYTE *OldBuffer = Buffer;
		int OldBufferSize = BufferSize;

		int Needed = DataSize - Remaining;

		BufferSize += Needed > BUFFER_SIZE_INCREMENT ? Needed : BUFFER_SIZE_INCREMENT;

		Buffer = new BYTE[BufferSize];

		memcpy(Buffer, OldBuffer, OldBufferSize);

		delete [] OldBuffer;
	}

	memcpy(Buffer + Position, Data, DataSize);

	Position += DataSize;
}

void CBuffer::Empty()
{
	delete [] Buffer;

	SetDefaults();
}

void *CBuffer::GetData()
{
	return Buffer;
}

int CBuffer::GetDataSize()
{
	return Position;
}

void CBuffer::SetDefaults()
{
	Buffer = NULL;

	BufferSize = 0;
	Position = 0;
}

// ----------------------------------------------------------------------------------------------------------------------------

int gl_max_texture_size = 0, gl_max_texture_max_anisotropy_ext = 0;

// ----------------------------------------------------------------------------------------------------------------------------

CTexture::CTexture()
{
	Texture = 0;
}

CTexture::~CTexture()
{
}

CTexture::operator GLuint ()
{
	return Texture;
}

bool CTexture::LoadTexture2D(char *FileName)
{
	CString DirectoryFileName = ModuleDirectory + FileName;

	int BPP;

	FIBITMAP *dib = GetBitmap(DirectoryFileName, Width, Height, BPP);

	if(dib == NULL)
	{
		ErrorLog.Append("Error loading texture " + DirectoryFileName + "!\r\n");
		return false;
	}

	GLenum Format = 0;

	if(BPP == 32) Format = GL_BGRA;
	if(BPP == 24) Format = GL_BGR;

	if(Format == 0)
	{
		ErrorLog.Append("Unsupported texture format (%s)!\r\n", FileName);
		FreeImage_Unload(dib);
		return false;
	}

	Destroy();

	glGenTextures(1, &Texture);

	glBindTexture(GL_TEXTURE_2D, Texture);

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	if(GLEW_EXT_texture_filter_anisotropic)
	{
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY_EXT, gl_max_texture_max_anisotropy_ext);
	}

	glTexParameteri(GL_TEXTURE_2D, GL_GENERATE_MIPMAP, GL_TRUE);

	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, Width, Height, 0, Format, GL_UNSIGNED_BYTE, FreeImage_GetBits(dib));

	glBindTexture(GL_TEXTURE_2D, 0);

	FreeImage_Unload(dib);

	return true;
}

bool CTexture::LoadTextureCubeMap(char **FileNames)
{
	int BPP;

	FIBITMAP *dib[6];

	bool Error = false;
	
	for(int i = 0; i < 6; i++)
	{
		CString DirectoryFileName = ModuleDirectory + FileNames[i];

		dib[i] = GetBitmap(DirectoryFileName, Width, Height, BPP);

		if(dib[i] == NULL)
		{
			ErrorLog.Append("Error loading texture " + DirectoryFileName + "!\r\n");
			Error = true;
		}
	}

	if(Error)
	{
		for(int i = 0; i < 6; i++)
		{
			FreeImage_Unload(dib[i]);
		}

		return false;
	}

	GLenum Format = 0;
	
	if(BPP == 32) Format = GL_BGRA;
	if(BPP == 24) Format = GL_BGR;

	if(Format == 0)
	{
		ErrorLog.Append("Unsupported texture format (%s)!\r\n", FileNames[5]);

		for(int i = 0; i < 6; i++)
		{
			FreeImage_Unload(dib[i]);
		}

		return false;
	}

	Destroy();

	glGenTextures(1, &Texture);

	glBindTexture(GL_TEXTURE_CUBE_MAP, Texture);

	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	if(GLEW_EXT_texture_filter_anisotropic)
	{
		glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MAX_ANISOTROPY_EXT, gl_max_texture_max_anisotropy_ext);
	}

	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_GENERATE_MIPMAP, GL_TRUE);

	for(int i = 0; i < 6; i++)
	{
		glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_X + i, 0, GL_RGBA8, Width, Height, 0, Format, GL_UNSIGNED_BYTE, FreeImage_GetBits(dib[i]));
	}

	glBindTexture(GL_TEXTURE_CUBE_MAP, 0);

	for(int i = 0; i < 6; i++)
	{
		FreeImage_Unload(dib[i]);
	}

	return true;
}

void CTexture::Destroy()
{
	glDeleteTextures(1, &Texture);
	Texture = 0;
}

FIBITMAP *CTexture::GetBitmap(char *FileName, int &Width, int &Height, int &BPP)
{
	FREE_IMAGE_FORMAT fif = FreeImage_GetFileType(FileName);

	if(fif == FIF_UNKNOWN)
	{
		fif = FreeImage_GetFIFFromFilename(FileName);
	}

	if(fif == FIF_UNKNOWN)
	{
		return NULL;
	}

	FIBITMAP *dib = NULL;

	if(FreeImage_FIFSupportsReading(fif))
	{
		dib = FreeImage_Load(fif, FileName);
	}

	if(dib != NULL)
	{
		int OriginalWidth = FreeImage_GetWidth(dib);
		int OriginalHeight = FreeImage_GetHeight(dib);

		Width = OriginalWidth;
		Height = OriginalHeight;

		if(Width == 0 || Height == 0)
		{
			FreeImage_Unload(dib);
			return NULL;
		}

		BPP = FreeImage_GetBPP(dib);

		if(Width > gl_max_texture_size) Width = gl_max_texture_size;
		if(Height > gl_max_texture_size) Height = gl_max_texture_size;

		if(!GLEW_ARB_texture_non_power_of_two)
		{
			Width = 1 << (int)floor((log((float)Width) / log(2.0f)) + 0.5f); 
			Height = 1 << (int)floor((log((float)Height) / log(2.0f)) + 0.5f);
		}

		if(Width != OriginalWidth || Height != OriginalHeight)
		{
			FIBITMAP *rdib = FreeImage_Rescale(dib, Width, Height, FILTER_BICUBIC);
			FreeImage_Unload(dib);
			dib = rdib;
		}
	}

	return dib;
}

// ----------------------------------------------------------------------------------------------------------------------------

CShaderProgram::CShaderProgram()
{
	SetDefaults();
}

CShaderProgram::~CShaderProgram()
{
}

CShaderProgram::operator GLuint ()
{
	return Program;
}

bool CShaderProgram::Load(char *VertexShaderFileName, char *FragmentShaderFileName)
{
	bool Error = false;

	Destroy();

	Error |= ((VertexShader = LoadShader(VertexShaderFileName, GL_VERTEX_SHADER)) == 0);
	Error |= ((FragmentShader = LoadShader(FragmentShaderFileName, GL_FRAGMENT_SHADER)) == 0);

	if(Error)
	{
		Destroy();
		return false;
	}

	Program = glCreateProgram();
	glAttachShader(Program, VertexShader);
	glAttachShader(Program, FragmentShader);
	glLinkProgram(Program);

	int LinkStatus;
	glGetProgramiv(Program, GL_LINK_STATUS, &LinkStatus);

	if(LinkStatus == GL_FALSE)
	{
		ErrorLog.Append("Error linking program (%s, %s)!\r\n", VertexShaderFileName, FragmentShaderFileName);

		int InfoLogLength = 0;
		glGetProgramiv(Program, GL_INFO_LOG_LENGTH, &InfoLogLength);
	
		if(InfoLogLength > 0)
		{
			char *InfoLog = new char[InfoLogLength];
			int CharsWritten  = 0;
			glGetProgramInfoLog(Program, InfoLogLength, &CharsWritten, InfoLog);
			ErrorLog.Append(InfoLog);
			delete [] InfoLog;
		}

		Destroy();

		return false;
	}

	return true;
}

void CShaderProgram::Destroy()
{
	glDetachShader(Program, VertexShader);
	glDetachShader(Program, FragmentShader);

	glDeleteShader(VertexShader);
	glDeleteShader(FragmentShader);

	glDeleteProgram(Program);

	delete [] UniformLocations;
	delete [] AttribLocations;

	SetDefaults();
}

GLuint CShaderProgram::LoadShader(char *FileName, GLenum Type)
{
	CString DirectoryFileName = ModuleDirectory + FileName;

	FILE *File;

	if(fopen_s(&File, DirectoryFileName, "rb") != 0)
	{
		ErrorLog.Append("Error loading file " + DirectoryFileName + "!\r\n");
		return 0;
	}

	fseek(File, 0, SEEK_END);
	long Size = ftell(File);
	fseek(File, 0, SEEK_SET);
	char *Source = new char[Size + 1];
	fread(Source, 1, Size, File);
	fclose(File);
	Source[Size] = 0;

	GLuint TerrainShader = glCreateShader(Type);

	glShaderSource(TerrainShader, 1, (const char**)&Source, NULL);
	delete [] Source;
	glCompileShader(TerrainShader);

	int CompileStatus;
	glGetShaderiv(TerrainShader, GL_COMPILE_STATUS, &CompileStatus);

	if(CompileStatus == GL_FALSE)
	{
		ErrorLog.Append("Error compiling shader %s!\r\n", FileName);

		int InfoLogLength = 0;
		glGetShaderiv(TerrainShader, GL_INFO_LOG_LENGTH, &InfoLogLength);
	
		if(InfoLogLength > 0)
		{
			char *InfoLog = new char[InfoLogLength];
			int CharsWritten  = 0;
			glGetShaderInfoLog(TerrainShader, InfoLogLength, &CharsWritten, InfoLog);
			ErrorLog.Append(InfoLog);
			delete [] InfoLog;
		}

		glDeleteShader(TerrainShader);

		return 0;
	}

	return TerrainShader;
}

void CShaderProgram::SetDefaults()
{
	VertexShader = 0;
	FragmentShader = 0;

	Program = 0;

	UniformLocations = NULL;
	AttribLocations = NULL;
}

// ----------------------------------------------------------------------------------------------------------------------------

CFrustum::CFrustum()
{
}

CFrustum::~CFrustum()
{
}

void CFrustum::Set(mat4x4 &ViewMatrixInverse, mat4x4 &ProjectionMatrixInverse)
{
	Set(-1.0f, -1.0f, 1.0f, 1.0f, ViewMatrixInverse, ProjectionMatrixInverse);
}

void CFrustum::Set(int AX, int AY, int BX, int BY, int WidthM1, int HeightM1, mat4x4 &ViewMatrixInverse, mat4x4 &ProjectionMatrixInverse)
{
	AY = HeightM1 - AY;
	BY = HeightM1 - BY;

	if(BX < AX)
	{
		int temp = AX;
		AX = BX;
		BX = temp;
	}

	if(BY < AY)
	{
		int temp = AY;
		AY = BY;
		BY = temp;
	}

	float ax = (float)AX / (float)WidthM1 * 2.0f - 1.0f;
	float ay = (float)AY / (float)HeightM1 * 2.0f - 1.0f;
	float bx = (float)BX / (float)WidthM1 * 2.0f - 1.0f;
	float by = (float)BY / (float)HeightM1 * 2.0f - 1.0f;

	Set(ax, ay, bx, by, ViewMatrixInverse, ProjectionMatrixInverse);
}

void CFrustum::Set(float AX, float AY, float BX, float BY, mat4x4 &ViewMatrixInverse, mat4x4 &ProjectionMatrixInverse)
{
	V[0] = vec4(AX, AY, -1.0f, 1.0f);
	V[1] = vec4(BX, AY, -1.0f, 1.0f);
	V[2] = vec4(BX, BY, -1.0f, 1.0f);
	V[3] = vec4(AX, BY, -1.0f, 1.0f);
	V[4] = vec4(AX, AY, 1.0f, 1.0f);
	V[5] = vec4(BX, AY, 1.0f, 1.0f);
	V[6] = vec4(BX, BY, 1.0f, 1.0f);
	V[7] = vec4(AX, BY, 1.0f, 1.0f);

	for(int i = 0; i < 8; i++)
	{
		V[i] = ViewMatrixInverse * (ProjectionMatrixInverse * V[i]);
		V[i] /= V[i].w;
	}

	N[0] = normalize(cross(*(vec3*)&V[2] - *(vec3*)&V[1], *(vec3*)&V[5] - *(vec3*)&V[1]));
	D[0] = -dot(N[0], *(vec3*)&V[1]);

	N[1] = normalize(cross(*(vec3*)&V[4] - *(vec3*)&V[0], *(vec3*)&V[3] - *(vec3*)&V[0]));
	D[1] = -dot(N[1], *(vec3*)&V[0]);

	N[2] = normalize(cross(*(vec3*)&V[1] - *(vec3*)&V[0], *(vec3*)&V[4] - *(vec3*)&V[0]));
	D[2] = -dot(N[2], *(vec3*)&V[0]);

	N[3] = normalize(cross(*(vec3*)&V[3] - *(vec3*)&V[2], *(vec3*)&V[6] - *(vec3*)&V[2]));
	D[3] = -dot(N[3], *(vec3*)&V[2]);

	N[4] = normalize(cross(*(vec3*)&V[3] - *(vec3*)&V[0], *(vec3*)&V[1] - *(vec3*)&V[0]));
	D[4] = -dot(N[4], *(vec3*)&V[0]);

	N[5] = normalize(cross(*(vec3*)&V[6] - *(vec3*)&V[5], *(vec3*)&V[4] - *(vec3*)&V[6]));
	D[5] = -dot(N[5], *(vec3*)&V[5]);
}

bool CFrustum::VertexInside(vec3 &Vertex)
{
	for(int i = 0; i < 6; i++)
	{
		if(dot(N[i], Vertex) + D[i] < 0.0f) return false;
	}

	return true;
}

// ----------------------------------------------------------------------------------------------------------------------------

CCamera::CCamera()
{
	X = vec3(1.0f, 0.0f, 0.0f);
	Y = vec3(0.0f, 1.0f, 0.0f);
	Z = vec3(0.0f, 0.0f, 1.0f);

	Position = vec3(0.0f, 0.0f, 5.0f);
	Reference = vec3(0.0f, 0.0f, 0.0f);

	CalculateViewMatrix();
}

CCamera::~CCamera()
{
}

void CCamera::Look(const vec3 &Position, const vec3 &Reference, bool RotateAroundReference)
{
	this->Position = Position;
	this->Reference = Reference;

	Z = normalize(Position - Reference);
	X = normalize(cross(vec3(0.0f, 1.0f, 0.0f), Z));
	Y = cross(Z, X);

	if(!RotateAroundReference)
	{
		this->Reference = this->Position;
		this->Position += Z * 0.05f;
	}

	CalculateViewMatrix();
}

void CCamera::Move(const vec3 &Movement)
{
	Position += Movement;
	Reference += Movement;

	CalculateViewMatrix();
}

vec3 CCamera::OnKeys(BYTE Keys, float FrameTime)
{
	float Speed = 5.0f;

	if(Keys & 0x40) Speed *= 2.0f;
	if(Keys & 0x80) Speed *= 0.5f;

	float Distance = Speed * FrameTime;

	vec3 Up(0.0f, 1.0f, 0.0f);
	vec3 Right = X;
	vec3 Forward = cross(Up, Right);

	Up *= Distance;
	Right *= Distance;
	Forward *= Distance;

	vec3 Movement;

	if(Keys & 0x01) Movement += Forward;
	if(Keys & 0x02) Movement -= Forward;
	if(Keys & 0x04) Movement -= Right;
	if(Keys & 0x08) Movement += Right;
	if(Keys & 0x10) Movement += Up;
	if(Keys & 0x20) Movement -= Up;

	return Movement;
}

void CCamera::OnMouseMove(int dx, int dy)
{
	float Sensitivity = 0.25f;

	Position -= Reference;

	if(dx != 0)
	{
		float DeltaX = (float)dx * Sensitivity;

		X = rotate(X, DeltaX, vec3(0.0f, 1.0f, 0.0f));
		Y = rotate(Y, DeltaX, vec3(0.0f, 1.0f, 0.0f));
		Z = rotate(Z, DeltaX, vec3(0.0f, 1.0f, 0.0f));
	}

	if(dy != 0)
	{
		float DeltaY = (float)dy * Sensitivity;

		Y = rotate(Y, DeltaY, X);
		Z = rotate(Z, DeltaY, X);

		if(Y.y < 0.0f)
		{
			Z = vec3(0.0f, Z.y > 0.0f ? 1.0f : -1.0f, 0.0f);
			Y = cross(Z, X);
		}
	}

	Position = Reference + Z * length(Position);

	CalculateViewMatrix();
}

void CCamera::OnMouseWheel(float zDelta)
{
	Position -= Reference;

	if(zDelta < 0 && length(Position) < 500.0f)
	{
		Position += Position * 0.1f;
	}

	if(zDelta > 0 && length(Position) > 0.05f)
	{
		Position -= Position * 0.1f;
	}

	Position += Reference;

	CalculateViewMatrix();
}

void CCamera::CalculateViewMatrix()
{
	ViewMatrix = mat4x4(X.x, Y.x, Z.x, 0.0f, X.y, Y.y, Z.y, 0.0f, X.z, Y.z, Z.z, 0.0f, -dot(X, Position), -dot(Y, Position), -dot(Z, Position), 1.0f);
	ViewMatrixInverse = inverse(ViewMatrix);

	Frustum.Set(ViewMatrixInverse, ProjectionMatrixInverse);
}

void CCamera::SetPerspectiveProjection(float Fovy, float Aspect, float N, float F)
{
	ProjectionMatrix = perspective(Fovy, Aspect, N, F);
	ProjectionMatrixInverse = inverse(ProjectionMatrix);

	Frustum.Set(ViewMatrixInverse, ProjectionMatrixInverse);
}

// ----------------------------------------------------------------------------------------------------------------------------

CCamera Camera;

// ----------------------------------------------------------------------------------------------------------------------------

CSelectedVerticesList::CSelectedVerticesList()
{
}

CSelectedVerticesList::~CSelectedVerticesList()
{
}

void CSelectedVerticesList::AddVertexIndex(int VertexIndex)
{
	if(VerticesIndicesFlags[VertexIndex] == 0)
	{
		VerticesIndices[VerticesIndicesCount++] = VertexIndex;
		VerticesIndicesFlags[VertexIndex] = 1;
	}
}

void CSelectedVerticesList::Create(int VerticesCount)
{
	VerticesIndices = new int[VerticesCount];
	VerticesIndicesFlags = new BYTE[VerticesCount];

	memset(VerticesIndicesFlags, 0, VerticesCount);

	this->VerticesCount = VerticesCount;

	VerticesIndicesCount = 0;
}

void CSelectedVerticesList::Destroy()
{
	delete [] VerticesIndices;
	delete [] VerticesIndicesFlags;
}

void CSelectedVerticesList::Empty()
{
	memset(VerticesIndicesFlags, 0, VerticesCount);
	VerticesIndicesCount = 0;
}

int CSelectedVerticesList::GetVertexIndex(int Index)
{
	return VerticesIndices[Index];
}

int CSelectedVerticesList::GetVerticesIndicesCount()
{
	return VerticesIndicesCount;
}

// ----------------------------------------------------------------------------------------------------------------------------

CTerrain::CTerrain()
{
}

CTerrain::~CTerrain()
{
}

bool CTerrain::Create(int Size)
{
	this->Size = Size;

	SizeP1 = Size + 1;
	SizeD2 = (float)Size / 2.0f;
	MSizeD2 = -SizeD2;
	ODSizeD2 = 1.0f / (float)Size / 2.0f;
	OMODSizeD2 = 1.0f - ODSizeD2;

	VerticesCount = SizeP1 * SizeP1;
	LinesIndicesCount = Size * Size * 8;
	QuadsIndicesCount = Size * Size * 4;

	Vertices = new vec3[VerticesCount];
	Normals = new vec3[VerticesCount];
	LinesIndices = new int[LinesIndicesCount];
	QuadsIndices = new int[QuadsIndicesCount];

	int i = 0;

	for(int z = 0; z <= Size; z++)
	{
		for(int x = 0; x <= Size; x++)
		{
			Vertices[i].x = (float)x - SizeD2;
			Vertices[i].y = 0.0f;
			Vertices[i].z = SizeD2 - (float)z;

			Normals[i].x = 0.0f;
			Normals[i].y = 1.0f;
			Normals[i].z = 0.0f;

			i++;
		}
	}

	i = 0;

	for(int z = 0; z < Size; z++)
	{
		for(int x = 0; x < Size; x++)
		{
			LinesIndices[i++] = GetVertexIndex(x, z);
			LinesIndices[i++] = GetVertexIndex(x + 1, z);
			LinesIndices[i++] = GetVertexIndex(x + 1, z);
			LinesIndices[i++] = GetVertexIndex(x + 1, z + 1);
			LinesIndices[i++] = GetVertexIndex(x + 1, z + 1);
			LinesIndices[i++] = GetVertexIndex(x, z + 1);
			LinesIndices[i++] = GetVertexIndex(x, z + 1);
			LinesIndices[i++] = GetVertexIndex(x, z);
		}
	}

	i = 0;

	for(int z = 0; z < Size; z++)
	{
		for(int x = 0; x < Size; x++)
		{
			QuadsIndices[i++] = GetVertexIndex(x, z);
			QuadsIndices[i++] = GetVertexIndex(x + 1, z);
			QuadsIndices[i++] = GetVertexIndex(x + 1, z + 1);
			QuadsIndices[i++] = GetVertexIndex(x, z + 1);
		}
	}

	glGenTextures(1, &HeightMapTexture);

	CopyVerticesToHeightMapTexture();

	SelectedVerticesIndices.Create(VerticesCount);

	return true;
}

bool CTerrain::Load(char *FileName)
{
	CString DirectoryFileName = ModuleDirectory + FileName;

	FILE *File;

	if(fopen_s(&File, DirectoryFileName, "rb") != 0)
	{
		return false;
	}

	int Size;

	fread(&Size, 4, 1, File);

	Create(Size);

	fread(Vertices, 12, VerticesCount, File);

	fclose(File);

	CalculateMinAndMaxHeights();
	CalculateNormals();
	CopyVerticesToHeightMapTexture();

	return true;
}

bool CTerrain::LoadHeightMapTexture(char *FileName, float ScaleHeight)
{
	CTexture HeightMapTexture;

	if(!HeightMapTexture.LoadTexture2D(FileName))
	{
		return false;
	}

	if(HeightMapTexture.Width != HeightMapTexture.Height)
	{
		HeightMapTexture.Destroy();
		ErrorLog.Append("Width and height of the height map texture must be equal!\r\n");
		return false;
	}

	Create(HeightMapTexture.Width - 1);

	vec3 *HeightMapTextureData = new vec3[VerticesCount];

	glBindTexture(GL_TEXTURE_2D, HeightMapTexture);
	glGetTexImage(GL_TEXTURE_2D, 0, GL_RGB, GL_FLOAT, HeightMapTextureData);
	glBindTexture(GL_TEXTURE_2D, 0);

	HeightMapTexture.Destroy();

	float MinY = HeightMapTextureData[0].y;

	for(int i = 1; i < VerticesCount; i++)
	{
		if(HeightMapTextureData[i].y < MinY) MinY = HeightMapTextureData[i].y;
	}

	for(int i = 0; i < VerticesCount; i++)
	{
		Vertices[i].y = (HeightMapTextureData[i].y - MinY) * 256.0f * ScaleHeight;
	}

	delete [] HeightMapTextureData;

	CalculateMinAndMaxHeights();
	CalculateNormals();
	CopyVerticesToHeightMapTexture();

	return true;
}

bool CTerrain::Save(char *FileName)
{
	CString DirectoryFileName = ModuleDirectory + FileName;

	FILE *File;

	if(fopen_s(&File, DirectoryFileName, "wb+") != 0)
	{
		return false;
	}

	fwrite(&Size, 4, 1, File);

	fwrite(Vertices, 12, VerticesCount, File);

	fclose(File);

	return true;
}

void CTerrain::RenderLines()
{
    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(3, GL_FLOAT, 12, Vertices);

	glDrawElements(GL_LINES, LinesIndicesCount, GL_UNSIGNED_INT, LinesIndices);

    glDisableClientState(GL_VERTEX_ARRAY);
}

void CTerrain::RenderQuads(bool NormalArray)
{
    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(3, GL_FLOAT, 12, Vertices);

	if(NormalArray)
	{
		glEnableClientState(GL_NORMAL_ARRAY);
	    glNormalPointer(GL_FLOAT, 12, Normals);
	}

	glDrawElements(GL_QUADS, QuadsIndicesCount, GL_UNSIGNED_INT, QuadsIndices);

    if(NormalArray)
	{
		glDisableClientState(GL_NORMAL_ARRAY);
	}

    glDisableClientState(GL_VERTEX_ARRAY);
}

void CTerrain::Destroy()
{
	delete [] Vertices;
	delete [] Normals;
	delete [] LinesIndices;
	delete [] QuadsIndices;

	glDeleteTextures(1, &HeightMapTexture);

	SelectedVerticesIndices.Destroy();
}

GLuint CTerrain::GetHeightMapTexture()
{
	return HeightMapTexture;
}

float CTerrain::GetMinHeight()
{
	return MinHeight;
}

float CTerrain::GetMaxHeight()
{
	return MaxHeight;
}

int CTerrain::GetSize()
{
	return Size;
}

float CTerrain::GetSizeD2()
{
	return SizeD2;
}

float CTerrain::GetMSizeD2()
{
	return MSizeD2;
}

float CTerrain::GetODSizeD2()
{
	return ODSizeD2;
}

float CTerrain::GetOMODSizeD2()
{
	return OMODSizeD2;
}

void CTerrain::CalculateMinAndMaxHeights()
{
	MinHeight = MaxHeight = Vertices[0].y;

	for(int i = 1; i < VerticesCount; i++)
	{
		if(Vertices[i].y < MinHeight) MinHeight = Vertices[i].y;
		if(Vertices[i].y > MaxHeight) MaxHeight = Vertices[i].y;
	}
}

void CTerrain::CalculateNormals()
{
	int i = 0;

	for(int z = 0; z <= Size; z++)
	{
		for(int x = 0; x <= Size; x++)
		{
			Normals[i++] = normalize(vec3(GetHeight(x - 1, z) - GetHeight(x + 1, z), 2.0f, GetHeight(x, z + 1) - GetHeight(x, z - 1)));
		}
	}
}

void CTerrain::CopyVerticesToHeightMapTexture()
{
	glBindTexture(GL_TEXTURE_2D, HeightMapTexture);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA16F, SizeP1, SizeP1, 0, GL_RGB, GL_FLOAT, Vertices);
	glBindTexture(GL_TEXTURE_2D, 0);
}

void CTerrain::Displace(float Displacement)
{
	if(SelectedVerticesIndices.GetVerticesIndicesCount() > 0)
	{
		for(int i = 0; i < SelectedVerticesIndices.GetVerticesIndicesCount(); i++)
		{
			Vertices[SelectedVerticesIndices.GetVertexIndex(i)].y += Displacement;
		}

		CalculateNormals();
	}
	else
	{
		for(int i = 0; i < VerticesCount; i++)
		{
			Vertices[i].y += Displacement;
		}
	}

	CalculateMinAndMaxHeights();
	CopyVerticesToHeightMapTexture();
}

float CTerrain::GetHeight(int X, int Z)
{
	return Vertices[GetVertexIndex(X < 0 ? 0 : X > Size ? Size : X, Z < 0 ? 0 : Z > Size ? Size : Z)].y;
}

float CTerrain::GetHeight(float X, float Z)
{
	Z = -Z;

	X += SizeD2;
	Z += SizeD2;

	float Size = (float)this->Size;

	if(X < 0.0f) X = 0.0f;
	if(X > Size) X = Size;
	if(Z < 0.0f) Z = 0.0f;
	if(Z > Size) Z = Size;

	int ix = (int)X;
	int iz = (int)Z;

	float fx = X - (float)ix;
	float fz = Z - (float)iz;

	float a = GetHeight(ix, iz);
	float b = GetHeight(ix + 1, iz);
	float c = GetHeight(ix, iz + 1);
	float d = GetHeight(ix + 1, iz + 1);

	float ab = a + (b - a) * fx;
	float cd = c + (d - c) * fx;

	return ab + (cd - ab) * fz;
}

void CTerrain::GetMinMax(mat4x4 &ViewMatrix, vec3 &Min, vec3 &Max)
{
	vec4 Vertex = ViewMatrix * vec4(Vertices[0], 1.0);

	Min.x = Max.x = Vertex.x;
	Min.y = Max.y = Vertex.y;
	Min.z = Max.z = Vertex.z;

	for(int i = 1; i < VerticesCount; i++)
	{
		Vertex = ViewMatrix * vec4(Vertices[i], 1.0);

		if(Vertex.x < Min.x) Min.x = Vertex.x;
		if(Vertex.y < Min.y) Min.y = Vertex.y;
		if(Vertex.z < Min.z) Min.z = Vertex.z;

		if(Vertex.x > Max.x) Max.x = Vertex.x;
		if(Vertex.y > Max.y) Max.y = Vertex.y;
		if(Vertex.z > Max.z) Max.z = Vertex.z;
	}
}

int CTerrain::GetVertexIndex(int X, int Z)
{
	return SizeP1 * Z + X;
}

void CTerrain::Randomize()
{
	if(SelectedVerticesIndices.GetVerticesIndicesCount() > 0)
	{
		for(int i = 0; i < SelectedVerticesIndices.GetVerticesIndicesCount(); i++)
		{
			Vertices[SelectedVerticesIndices.GetVertexIndex(i)].y += (float)rand() / (float)RAND_MAX - 0.5f;
		}
	}
	else
	{
		for(int i = 0; i < VerticesCount; i++)
		{
			Vertices[i].y += (float)rand() / (float)RAND_MAX - 0.5f;
		}
	}

	CalculateMinAndMaxHeights();
	CalculateNormals();
	CopyVerticesToHeightMapTexture();
}

float SelectionBoxVertices[] = {
	-0.125f, -0.125f,  0.125f,
	 0.125f, -0.125f,  0.125f,
	 0.125f,  0.125f,  0.125f,
	-0.125f,  0.125f,  0.125f,
	-0.125f, -0.125f, -0.125f,
	 0.125f, -0.125f, -0.125f,
	 0.125f,  0.125f, -0.125f,
	-0.125f,  0.125f, -0.125f
};

int SelectionBoxLinesIndices[] = {
	0, 1, 1, 2, 2, 3, 3, 0,
	4, 5, 5, 6, 6, 7, 7, 4,
	0, 4, 1, 5, 2, 6, 3, 7
};

void CTerrain::RenderSelectionBoxAtVertex(int VertexIndex, mat4x4 &ViewMatrix)
{
    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(3, GL_FLOAT, 12, SelectionBoxVertices);

	glLoadMatrixf(&ViewMatrix);
	glTranslatef(Vertices[VertexIndex].x, Vertices[VertexIndex].y, Vertices[VertexIndex].z);

	glDrawElements(GL_LINES, 24, GL_UNSIGNED_INT, SelectionBoxLinesIndices);

    glDisableClientState(GL_VERTEX_ARRAY);
}

void CTerrain::RenderSelectionBoxesAtSelectedVertices(mat4x4 &ViewMatrix)
{
	if(SelectedVerticesIndices.GetVerticesIndicesCount() > 0)
	{
		glMatrixMode(GL_MODELVIEW);

		for(int i = 0; i < SelectedVerticesIndices.GetVerticesIndicesCount(); i++)
		{
			RenderSelectionBoxAtVertex(SelectedVerticesIndices.GetVertexIndex(i), ViewMatrix);
		}

		glLoadMatrixf(&ViewMatrix);
	}
}

void CTerrain::SelectVertices(CFrustum Frustum)
{
	for(int i = 0; i < VerticesCount; i++)
	{
		if(Frustum.VertexInside(Vertices[i]))
		{
			SelectedVerticesIndices.AddVertexIndex(i);
		}
	}
}

void CTerrain::Smooth()
{
	if(SelectedVerticesIndices.GetVerticesIndicesCount() > 0)
	{
		float *NewHeights = new float[SelectedVerticesIndices.GetVerticesIndicesCount()];

		for(int i = 0; i < SelectedVerticesIndices.GetVerticesIndicesCount(); i++)
		{
			int VertexIndex = SelectedVerticesIndices.GetVertexIndex(i);

			int x = (int)(Vertices[VertexIndex].x + SizeD2);
			int z = (int)(SizeD2 - Vertices[VertexIndex].z);

			NewHeights[i] = 0.0f;

			NewHeights[i] += GetHeight(x - 1, z + 1) + GetHeight(x, z + 1) * 2 + GetHeight(x + 1, z + 1);
			NewHeights[i] += GetHeight(x - 1, z) * 2 + GetHeight(x, z) * 3 + GetHeight(x + 1, z) * 2;
			NewHeights[i] += GetHeight(x - 1, z - 1) + GetHeight(x, z - 1) * 2 + GetHeight(x + 1, z - 1);

			NewHeights[i] /= 15.0f;
		}

		for(int i = 0; i < SelectedVerticesIndices.GetVerticesIndicesCount(); i++)
		{
			Vertices[SelectedVerticesIndices.GetVertexIndex(i)].y = NewHeights[i];
		}

		delete [] NewHeights;
	}
	else
	{
		float *NewHeights = new float[VerticesCount];

		int i = 0;

		for(int z = 0; z <= Size; z++)
		{
			for(int x = 0; x <= Size; x++)
			{
				NewHeights[i] = 0.0f;

				NewHeights[i] += GetHeight(x - 1, z + 1) + GetHeight(x, z + 1) * 2 + GetHeight(x + 1, z + 1);
				NewHeights[i] += GetHeight(x - 1, z) * 2 + GetHeight(x, z) * 3 + GetHeight(x + 1, z) * 2;
				NewHeights[i] += GetHeight(x - 1, z - 1) + GetHeight(x, z - 1) * 2 + GetHeight(x + 1, z - 1);

				NewHeights[i] /= 15.0f;

				i++;
			}
		}

		for(int i = 0; i < VerticesCount; i++)
		{
			Vertices[i].y = NewHeights[i];
		}

		delete [] NewHeights;
	}

	CalculateMinAndMaxHeights();
	CalculateNormals();
	CopyVerticesToHeightMapTexture();
}

void CTerrain::UnselectAllVertices()
{
	SelectedVerticesIndices.Empty();
}

// ----------------------------------------------------------------------------------------------------------------------------

COpenGLRenderer::COpenGLRenderer()
{
	LightAngle = 0.0f; 

	AX = AY = BX = BY = -1;

	RenderLines = true;
	RenderWater = true;
	DisplayMap = true;
	DisplayShadowMap = false;
}

COpenGLRenderer::~COpenGLRenderer()
{
}

bool COpenGLRenderer::Init()
{
	// ------------------------------------------------------------------------------------------------------------------------

	bool Error = false;

	// ------------------------------------------------------------------------------------------------------------------------

	if(!GLEW_ARB_texture_non_power_of_two)
	{
		ErrorLog.Append("GL_ARB_texture_non_power_of_two not supported!\r\n");
		Error = true;
	}

	if(!GLEW_ARB_texture_float)
	{
		ErrorLog.Append("GL_ARB_texture_float not supported!\r\n");
		Error = true;
	}

	if(!GLEW_EXT_framebuffer_object)
	{
		ErrorLog.Append("GL_EXT_framebuffer_object not supported!\r\n");
		Error = true;
	}

	// ------------------------------------------------------------------------------------------------------------------------

	Error |= !TerrainShader.Load("terrain.vs", "terrain.fs");
	Error |= !WaterShader.Load("water.vs", "water.fs");
	Error |= !MapShader.Load("map.vs", "map.fs");

	// ------------------------------------------------------------------------------------------------------------------------

	int terrain = 2;

	switch(terrain)
	{
		case 0: Error = !Terrain.Load("terrain0.xyz"); break;
		case 1: Error = !Terrain.LoadHeightMapTexture("terrain1.jpg", 0.0625f); break;
		case 2: Error = !Terrain.LoadHeightMapTexture("terrain2.jpg", 0.5f); break;
		case 3: Error = !Terrain.LoadHeightMapTexture("terrain3.jpg", 0.5f); break;
	}

	// ------------------------------------------------------------------------------------------------------------------------

	if(Error)
	{
		return false;
	}

	// ------------------------------------------------------------------------------------------------------------------------

	if(terrain == 1)
	{
		Terrain.Displace(-2.5f);
	}

	if(terrain == 2 || terrain == 3)
	{
		Terrain.Smooth();
		Terrain.Smooth();
	}
	
	if(terrain == 3)
	{
		Terrain.Displace(-7.0f);
	}

	// ------------------------------------------------------------------------------------------------------------------------

	TerrainShader.UniformLocations = new GLuint[2];
	TerrainShader.UniformLocations[0] = glGetUniformLocation(TerrainShader, "ShadowMatrix");
	TerrainShader.UniformLocations[1] = glGetUniformLocation(TerrainShader, "LightDirection");

	MapShader.UniformLocations = new GLuint[1];
	MapShader.UniformLocations[0] = glGetUniformLocation(MapShader, "MaxHeightD2");

	// ------------------------------------------------------------------------------------------------------------------------

	glUseProgram(TerrainShader);
	glUniform1i(glGetUniformLocation(TerrainShader, "ShadowMap"), 0);
	glUniform1i(glGetUniformLocation(TerrainShader, "RotationTexture"), 1);
	glUniform1f(glGetUniformLocation(TerrainShader, "Scale"), 1.0f / 64.0f);
	glUniform1f(glGetUniformLocation(TerrainShader, "Radius"), 1.0f / 1024.0f);
	glUseProgram(0);

	glUseProgram(MapShader);
	glUniform1f(MapShader.UniformLocations[0], Terrain.GetMaxHeight() / 2.0f);
	glUseProgram(0);

	// ------------------------------------------------------------------------------------------------------------------------

	ShadowMapSize = SHADOW_MAP_SIZE > gl_max_texture_size ? gl_max_texture_size : SHADOW_MAP_SIZE;

    glGenTextures(1, &ShadowMap);
    glBindTexture(GL_TEXTURE_2D, ShadowMap);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_DEPTH_COMPONENT24, ShadowMapSize, ShadowMapSize, 0, GL_DEPTH_COMPONENT, GL_FLOAT, NULL);
    glBindTexture(GL_TEXTURE_2D, 0);

	// ------------------------------------------------------------------------------------------------------------------------

	srand(GetTickCount());

	// ------------------------------------------------------------------------------------------------------------------------

	vec4 *RotationTextureData = new vec4[4096];

	float Angle = 3.14f * 2.0f * (float)rand() / (float)RAND_MAX;

	for(int i = 0; i < 4096; i++)
	{
		RotationTextureData[i].x = cos(Angle);
		RotationTextureData[i].y = sin(Angle);
		RotationTextureData[i].z = -RotationTextureData[i].y;
		RotationTextureData[i].w = RotationTextureData[i].x;

		RotationTextureData[i] *= 0.5f;
		RotationTextureData[i] += 0.5f;

		Angle += 3.14f * 2.0f * (float)rand() / (float)RAND_MAX;
	}

	glGenTextures(1, &RotationTexture);
	glBindTexture(GL_TEXTURE_2D, RotationTexture);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, 64, 64, 0, GL_RGBA, GL_FLOAT, RotationTextureData);
	glBindTexture(GL_TEXTURE_2D, 0);

	delete [] RotationTextureData;

	// ------------------------------------------------------------------------------------------------------------------------

	glGenFramebuffersEXT(1, &FBO);
	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, FBO);
	glDrawBuffers(0, NULL); glReadBuffer(GL_NONE);
	glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_TEXTURE_2D, ShadowMap, 0);
	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);

	// ------------------------------------------------------------------------------------------------------------------------

	MoveLight(90.0f);

	CalculateShadowMatrix();

	RenderShadowMap();

	// ------------------------------------------------------------------------------------------------------------------------

	Camera.Look(vec3(0.0f, Terrain.GetSizeD2(), Terrain.GetSizeD2()), vec3(0.0f, 1.75f, 0.0f), true);

	// ------------------------------------------------------------------------------------------------------------------------

	CheckCameraTerrainPosition();

	// ------------------------------------------------------------------------------------------------------------------------

	SetText();

	// ------------------------------------------------------------------------------------------------------------------------

	return true;

	// ------------------------------------------------------------------------------------------------------------------------
}

void COpenGLRenderer::Render(float FrameTime)
{
	// ------------------------------------------------------------------------------------------------------------------------

	glViewport(0, 0, Width, Height);

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glEnable(GL_DEPTH_TEST);

	glMatrixMode(GL_PROJECTION);
	glLoadMatrixf(&Camera.ProjectionMatrix);

	glMatrixMode(GL_MODELVIEW);
	glLoadMatrixf(&Camera.ViewMatrix);

	// ------------------------------------------------------------------------------------------------------------------------

	glColor3f(1.0f, 0.5f, 0.0f);

	glBegin(GL_POINTS);
		glVertex3fv(&LightPosition);
	glEnd();

	vec3 Reference = Camera.Reference - vec3(0.0f, 1.75f, 0.0f);

	glBegin(GL_LINES);
		glVertex3f(Reference.x + 0.125f, Reference.y, Reference.z);
		glVertex3f(Reference.x - 0.125f, Reference.y, Reference.z);
		glVertex3f(Reference.x, Reference.y - 0.125f, Reference.z);
		glVertex3f(Reference.x, Reference.y + 0.125f, Reference.z);
		glVertex3f(Reference.x, Reference.y, Reference.z - 0.125f);
		glVertex3f(Reference.x, Reference.y, Reference.z + 0.125f);
	glEnd();

	// ------------------------------------------------------------------------------------------------------------------------

	if(RenderLines)
	{
		glColor3f(0.125f, 0.125f, 0.125f);

		Terrain.RenderLines();
	}

	// ------------------------------------------------------------------------------------------------------------------------

	glEnable(GL_CULL_FACE);

	glColor3f(1.0f, 1.0f, 1.0f);

	glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, ShadowMap);
	glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_2D, RotationTexture);

	glUseProgram(TerrainShader);

	Terrain.RenderQuads(true);

	glUseProgram(0);

	glActiveTexture(GL_TEXTURE1); glBindTexture(GL_TEXTURE_2D, 0);
	glActiveTexture(GL_TEXTURE0); glBindTexture(GL_TEXTURE_2D, 0);

	glDisable(GL_CULL_FACE);

	// ------------------------------------------------------------------------------------------------------------------------

	glColor3f(0.0f, 1.0f, 0.0f);

	Terrain.RenderSelectionBoxesAtSelectedVertices(Camera.ViewMatrix);

	// ------------------------------------------------------------------------------------------------------------------------

	if(RenderWater)
	{
		glEnable(GL_BLEND);
		
		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

		glColor4f(0.5f, 0.75f, 1.0f, 0.25f);
		
		glBindTexture(GL_TEXTURE_2D, Terrain.GetHeightMapTexture());

		glUseProgram(WaterShader);

		glBegin(GL_QUADS);
			glTexCoord2f(Terrain.GetODSizeD2(), Terrain.GetODSizeD2()); glVertex3f(Terrain.GetMSizeD2(), 0.0f, Terrain.GetSizeD2());
			glTexCoord2f(Terrain.GetOMODSizeD2(), Terrain.GetODSizeD2()); glVertex3f(Terrain.GetSizeD2(), 0.0f, Terrain.GetSizeD2());
			glTexCoord2f(Terrain.GetOMODSizeD2(), Terrain.GetOMODSizeD2()); glVertex3f(Terrain.GetSizeD2(), 0.0f, Terrain.GetMSizeD2());
			glTexCoord2f(Terrain.GetODSizeD2(), Terrain.GetOMODSizeD2()); glVertex3f(Terrain.GetMSizeD2(), 0.0f, Terrain.GetMSizeD2());
		glEnd();

		glUseProgram(0);
		
		glBindTexture(GL_TEXTURE_2D, 0);

		glDisable(GL_BLEND);
	}

	// ------------------------------------------------------------------------------------------------------------------------

	glDisable(GL_DEPTH_TEST);

	// ------------------------------------------------------------------------------------------------------------------------

	if(AX >= 0 && AX <= WidthM1 && AY >= 0 && AY <= HeightM1 && BX >= 0 && BX <= WidthM1 && BY >= 0 && BY <= HeightM1)
	{
		glMatrixMode(GL_PROJECTION);
		glLoadMatrixf(&OrthoMatrix);

		glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();

		float ax = (float)AX;
		float ay = (float)AY;
		float bx = (float)BX;
		float by = (float)BY;

		glColor3f(0.0f, 1.0f, 0.0f);

		glBegin(GL_LINES);
			glVertex2f(ax, ay); glVertex2f(bx, ay);
			glVertex2f(bx, ay); glVertex2f(bx, by);
			glVertex2f(bx, by); glVertex2f(ax, by);
			glVertex2f(ax, by); glVertex2f(ax, ay);
		glEnd();
	}

	// ------------------------------------------------------------------------------------------------------------------------

	if(DisplayMap)
	{
		glViewport(16, 16, 256, 256);

		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();

		glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();

		glColor3f(1.0f, 1.0f, 1.0f);

		glEnable(GL_TEXTURE_2D);

		glBindTexture(GL_TEXTURE_2D, Terrain.GetHeightMapTexture());

		glUseProgram(MapShader);

		glBegin(GL_QUADS);
			glTexCoord2f(Terrain.GetODSizeD2(), Terrain.GetODSizeD2()); glVertex2f(-1.0f, -1.0f);
			glTexCoord2f(Terrain.GetOMODSizeD2(), Terrain.GetODSizeD2()); glVertex2f(1.0f, -1.0f);
			glTexCoord2f(Terrain.GetOMODSizeD2(), Terrain.GetOMODSizeD2()); glVertex2f(1.0f, 1.0f);
			glTexCoord2f(Terrain.GetODSizeD2(), Terrain.GetOMODSizeD2()); glVertex2f(-1.0f, 1.0f);
		glEnd();

		glUseProgram(0);

		glBindTexture(GL_TEXTURE_2D, 0);

		glDisable(GL_TEXTURE_2D);

		vec2 Position = (vec2(Camera.Reference.x, -Camera.Reference.z) + Terrain.GetSizeD2()) / (float)Terrain.GetSize() * 2.0f - 1.0f;

		glColor3f(0.0f, 0.0f, 0.0f);

		glBegin(GL_LINES);
			glVertex2f(Position.x - 0.03125f, Position.y); glVertex2f(Position.x + 0.03125f, Position.y);
			glVertex2f(Position.x, Position.y - 0.03125f); glVertex2f(Position.x, Position.y + 0.03125f);
		glEnd();
	}

	// ------------------------------------------------------------------------------------------------------------------------

	if(DisplayShadowMap)
	{
		glViewport(Width - 1 - 16 - 256, 16, 256, 256);

		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();

		glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();

		glColor3f(1.0f, 1.0f, 1.0f);

		glEnable(GL_TEXTURE_2D);

		glBindTexture(GL_TEXTURE_2D, ShadowMap);

		glBegin(GL_QUADS);
			glTexCoord2f(0.0f, 0.0f); glVertex2f(-1.0f, -1.0f);
			glTexCoord2f(1.0f, 0.0f); glVertex2f(1.0f, -1.0f);
			glTexCoord2f(1.0f, 1.0f); glVertex2f(1.0f, 1.0f);
			glTexCoord2f(0.0f, 1.0f); glVertex2f(-1.0f, 1.0f);
		glEnd();

		glBindTexture(GL_TEXTURE_2D, 0);

		glDisable(GL_TEXTURE_2D);
	}

	// ------------------------------------------------------------------------------------------------------------------------
}

void COpenGLRenderer::Resize(int Width, int Height)
{
	this->Width = Width;
	this->Height = Height;

	WidthM1 = Width - 1;
	HeightM1 = Height - 1;

	Camera.SetPerspectiveProjection(45.0f, (float)Width / (float)Height, 0.125f, 1024.0f);

	OrthoMatrix = ortho(0.0f, (float)WidthM1, (float)HeightM1, 0.0f, 0.0f, 1.0f);
}

void COpenGLRenderer::Destroy()
{
	TerrainShader.Destroy();
	WaterShader.Destroy();
	MapShader.Destroy();

	Terrain.Destroy();

	glDeleteTextures(1, &ShadowMap);
	glDeleteTextures(1, &RotationTexture);

	if(GLEW_EXT_framebuffer_object)
	{
		glDeleteFramebuffersEXT(1, &FBO);
	}
}

void COpenGLRenderer::SetText()
{
	Text.Set("MinHeight = %f, MaxHeight = %f", Terrain.GetMinHeight(), Terrain.GetMaxHeight());
}

void COpenGLRenderer::CalculateShadowMatrix()
{
	LightViewMatrix = look(LightPosition, vec3(0.0), vec3(0.0f, 1.0f, 0.0f));

	vec3 Min, Max;

	Terrain.GetMinMax(LightViewMatrix, Min, Max);

	LightProjectionMatrix = ortho(Min.x, Max.x, Min.y, Max.y, -Max.z, -Min.z);

	ShadowMatrix = BiasMatrix * LightProjectionMatrix * LightViewMatrix;

	glUseProgram(TerrainShader);
	glUniformMatrix4fv(TerrainShader.UniformLocations[0], 1, GL_FALSE, &ShadowMatrix);
	glUseProgram(0);
}

void COpenGLRenderer::CheckCameraTerrainPosition()
{
	float TerrainSizeD2 = Terrain.GetSizeD2();

	if(Camera.Reference.x < -TerrainSizeD2) Camera.Move(vec3(-TerrainSizeD2 - Camera.Reference.x, 0.0f, 0.0f));
	if(Camera.Reference.x > TerrainSizeD2) Camera.Move(vec3(TerrainSizeD2 - Camera.Reference.x, 0.0f, 0.0f));
	if(Camera.Reference.z < -TerrainSizeD2) Camera.Move(vec3(0.0f, 0.0f, -TerrainSizeD2 - Camera.Reference.z));
	if(Camera.Reference.z > TerrainSizeD2) Camera.Move(vec3(0.0f, 0.0f, TerrainSizeD2 - Camera.Reference.z));

	Camera.Move(vec3(0.0f, Terrain.GetHeight(Camera.Reference.x, Camera.Reference.z) + 1.75f - Camera.Reference.y, 0.0f));
}

void COpenGLRenderer::MoveLight(float Angle)
{
	LightAngle += Angle;

	LightPosition = rotate(vec3((float)Terrain.GetSize(), 0.0f, 0.0f), -LightAngle, vec3(0.0f, 1.0f, -1.0f));

	LightDirection = normalize(LightPosition);

	glUseProgram(TerrainShader);
	glUniform3fv(TerrainShader.UniformLocations[1], 1, &LightDirection);
	glUseProgram(0);
}

void COpenGLRenderer::RenderShadowMap()
{
	glViewport(0, 0, ShadowMapSize, ShadowMapSize);

	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, FBO);

	glClear(GL_DEPTH_BUFFER_BIT);

	glMatrixMode(GL_PROJECTION);
	glLoadMatrixf(&LightProjectionMatrix);

	glMatrixMode(GL_MODELVIEW);
	glLoadMatrixf(&LightViewMatrix);

	glEnable(GL_DEPTH_TEST);

	Terrain.RenderQuads(false);

	glDisable(GL_DEPTH_TEST);

	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
}

void COpenGLRenderer::OnKeyDown(UINT Key)
{
	bool TerrainChanged = false, LightChanged = false;

	switch(Key)
	{
		case VK_F1:
			Terrain.Smooth();
			TerrainChanged = true;
			break;

		case VK_F2:
			Terrain.Randomize();
			TerrainChanged = true;
			break;

		case VK_F3:
			DisplayMap = !DisplayMap;
			break;

		case VK_F4:
			DisplayShadowMap = !DisplayShadowMap;
			break;

		case VK_F5:
			Terrain.Save("terrain-saved.xyz");
			break;

		case '1':
			RenderLines = !RenderLines;
			break;

		case '2':
			RenderWater = !RenderWater;
			break;

		case 'R':
			Terrain.Displace(1.0f);
			TerrainChanged = true;
			break;

		case 'F':
			Terrain.Displace(-1.0f);
			TerrainChanged = true;
			break;

		case 'N':
			if(GetKeyState(VK_CONTROL) & 0x80)
			{
				Terrain.Destroy();
				Terrain.Create(128);
				TerrainChanged = true;
			}
			break;

		case VK_ADD:
			MoveLight(7.5f);
			LightChanged = true;
			break;

		case VK_SUBTRACT:
			MoveLight(-7.5f);
			LightChanged = true;
			break;
	}

	if(TerrainChanged)
	{
		glUseProgram(MapShader);
		glUniform1f(MapShader.UniformLocations[0], Terrain.GetMaxHeight() / 2.0f);
		glUseProgram(0);

		CalculateShadowMatrix();
		RenderShadowMap();
		CheckCameraTerrainPosition();
		SetText();
	}

	if(LightChanged)
	{
		CalculateShadowMatrix();
		RenderShadowMap();
	}
}

void COpenGLRenderer::OnLButtonDown(int X, int Y)
{
	AX = BX = X;
	AY = BY = Y;
}

void COpenGLRenderer::OnLButtonUp(int X, int Y)
{
	if(!(GetKeyState(VK_CONTROL) & 0x80))
	{
		Terrain.UnselectAllVertices();
	}

	if(AX != BX && AY != BY)
	{
		CFrustum Frustum;

		Frustum.Set(AX, AY, BX, BY, WidthM1, HeightM1, Camera.ViewMatrixInverse, Camera.ProjectionMatrixInverse);

		Terrain.SelectVertices(Frustum);
	}

	AX = AY = BX = BY = -1;
}

void COpenGLRenderer::OnMouseMove(int X, int Y)
{
	if(GetKeyState(VK_LBUTTON) & 0x80)
	{
		BX = X;
		BY = Y;
	}
}

// ----------------------------------------------------------------------------------------------------------------------------

COpenGLRenderer OpenGLRenderer;

// ----------------------------------------------------------------------------------------------------------------------------

CString ModuleDirectory, ErrorLog;

// ----------------------------------------------------------------------------------------------------------------------------

void GetModuleDirectory()
{
	char *moduledirectory = new char[256];
	GetModuleFileName(GetModuleHandle(NULL), moduledirectory, 256);
	*(strrchr(moduledirectory, '\\') + 1) = 0;
	ModuleDirectory = moduledirectory;
	delete [] moduledirectory;
}

// ----------------------------------------------------------------------------------------------------------------------------

COpenGLView::COpenGLView()
{
}

COpenGLView::~COpenGLView()
{
}

bool COpenGLView::Init(HINSTANCE hInstance, char *Title, int Width, int Height, int Samples)
{
	this->Title = Title;
	this->Width = Width;
	this->Height = Height;

	WNDCLASSEX WndClassEx;

	memset(&WndClassEx, 0, sizeof(WNDCLASSEX));

	WndClassEx.cbSize = sizeof(WNDCLASSEX);
	WndClassEx.style = CS_OWNDC | CS_HREDRAW | CS_VREDRAW;
	WndClassEx.lpfnWndProc = WndProc;
	WndClassEx.hInstance = hInstance;
	WndClassEx.hIcon = LoadIcon(NULL, IDI_APPLICATION);
	WndClassEx.hIconSm = LoadIcon(NULL, IDI_APPLICATION);
	WndClassEx.hCursor = LoadCursor(NULL, IDC_ARROW);
	WndClassEx.lpszClassName = "Win32OpenGLWindowClass";

	if(RegisterClassEx(&WndClassEx) == 0)
	{
		ErrorLog.Set("RegisterClassEx failed!");
		return false;
	}

	DWORD Style = WS_OVERLAPPEDWINDOW | WS_CLIPSIBLINGS | WS_CLIPCHILDREN;

	hWnd = CreateWindowEx(WS_EX_APPWINDOW, WndClassEx.lpszClassName, Title, Style, 0, 0, Width, Height, NULL, NULL, hInstance, NULL);

	if(hWnd == NULL)
	{
		ErrorLog.Set("CreateWindowEx failed!");
		return false;
	}

	HDC hDC = GetDC(hWnd);

	if(hDC == NULL)
	{
		ErrorLog.Set("GetDC failed!");
		return false;
	}

	PIXELFORMATDESCRIPTOR pfd;

	memset(&pfd, 0, sizeof(PIXELFORMATDESCRIPTOR));

	pfd.nSize = sizeof(PIXELFORMATDESCRIPTOR);
	pfd.nVersion = 1;
	pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
	pfd.iPixelType = PFD_TYPE_RGBA;
	pfd.cColorBits = 32;
	pfd.cDepthBits = 24;
	pfd.iLayerType = PFD_MAIN_PLANE;

	int PixelFormat = ChoosePixelFormat(hDC, &pfd);

	if(PixelFormat == 0)
	{
		ErrorLog.Set("ChoosePixelFormat failed!");
		return false;
	}

	static int MSAAPixelFormat = 0;

	if(SetPixelFormat(hDC, MSAAPixelFormat == 0 ? PixelFormat : MSAAPixelFormat, &pfd) == FALSE)
	{
		ErrorLog.Set("SetPixelFormat failed!");
		return false;
	}

	hGLRC = wglCreateContext(hDC);

	if(hGLRC == NULL)
	{
		ErrorLog.Set("wglCreateContext failed!");
		return false;
	}

	if(wglMakeCurrent(hDC, hGLRC) == FALSE)
	{
		ErrorLog.Set("wglMakeCurrent failed!");
		return false;
	}

	if(glewInit() != GLEW_OK)
	{
		ErrorLog.Set("glewInit failed!");
		return false;
	}

	if(!GLEW_VERSION_2_1)
	{
		ErrorLog.Set("OpenGL 2.1 not supported!");
		return false;
	}

	if(MSAAPixelFormat == 0 && Samples > 0)
	{
		if(GLEW_ARB_multisample && WGLEW_ARB_pixel_format)
		{
			while(Samples > 0)
			{
				UINT NumFormats = 0;

				int PFAttribs[] =
				{
					WGL_DRAW_TO_WINDOW_ARB, GL_TRUE,
					WGL_SUPPORT_OPENGL_ARB, GL_TRUE,
					WGL_DOUBLE_BUFFER_ARB, GL_TRUE,
					WGL_PIXEL_TYPE_ARB, WGL_TYPE_RGBA_ARB,
					WGL_COLOR_BITS_ARB, 32,
					WGL_DEPTH_BITS_ARB, 24,
					WGL_ACCELERATION_ARB, WGL_FULL_ACCELERATION_ARB,
					WGL_SAMPLE_BUFFERS_ARB, GL_TRUE,
					WGL_SAMPLES_ARB, Samples,
					0
				};

				if(wglChoosePixelFormatARB(hDC, PFAttribs, NULL, 1, &MSAAPixelFormat, &NumFormats) == TRUE && NumFormats > 0) break;

				Samples--;
			}

			wglDeleteContext(hGLRC);
			DestroyWindow(hWnd);
			UnregisterClass(WndClassEx.lpszClassName, hInstance);

			return Init(hInstance, Title, Width, Height, Samples);
		}
		else
		{
			Samples = 0;
		}
	}

	this->Samples = Samples;

	GetModuleDirectory();

	glGetIntegerv(GL_MAX_TEXTURE_SIZE, &gl_max_texture_size);

	if(GLEW_EXT_texture_filter_anisotropic)
	{
		glGetIntegerv(GL_MAX_TEXTURE_MAX_ANISOTROPY_EXT, &gl_max_texture_max_anisotropy_ext);
	}

	if(WGLEW_EXT_swap_control)
	{
		wglSwapIntervalEXT(0);
	}

	return OpenGLRenderer.Init();
}

void COpenGLView::Show(bool Maximized)
{
	RECT dRect, wRect, cRect;

	GetWindowRect(GetDesktopWindow(), &dRect);
	GetWindowRect(hWnd, &wRect);
	GetClientRect(hWnd, &cRect);

	wRect.right += Width - cRect.right;
	wRect.bottom += Height - cRect.bottom;
	wRect.right -= wRect.left;
	wRect.bottom -= wRect.top;
	wRect.left = dRect.right / 2 - wRect.right / 2;
	wRect.top = dRect.bottom / 2 - wRect.bottom / 2;

	MoveWindow(hWnd, wRect.left, wRect.top, wRect.right, wRect.bottom, FALSE);

	ShowWindow(hWnd, Maximized ? SW_SHOWMAXIMIZED : SW_SHOWNORMAL);
}

void COpenGLView::MessageLoop()
{
	MSG Msg;

	while(GetMessage(&Msg, NULL, 0, 0) > 0)
	{
		TranslateMessage(&Msg);
		DispatchMessage(&Msg);
	}
}

void COpenGLView::Destroy()
{
	if(GLEW_VERSION_2_1)
	{
		OpenGLRenderer.Destroy();
	}

	wglDeleteContext(hGLRC);
	DestroyWindow(hWnd);
}

void COpenGLView::OnKeyDown(UINT Key)
{
	OpenGLRenderer.OnKeyDown(Key);
}

void COpenGLView::OnLButtonDown(int X, int Y)
{
	OpenGLRenderer.OnLButtonDown(X, Y);

	LastX = X;
	LastY = Y;
}

void COpenGLView::OnLButtonUp(int X, int Y)
{
	OpenGLRenderer.OnLButtonUp(X, Y);

	LastX = X;
	LastY = Y;
}

void COpenGLView::OnMouseMove(int X, int Y)
{
	if(GetKeyState(VK_RBUTTON) & 0x80)
	{
		Camera.OnMouseMove(LastX - X, LastY - Y);
	}

	OpenGLRenderer.OnMouseMove(X, Y);

	LastX = X;
	LastY = Y;
}

void COpenGLView::OnMouseWheel(short zDelta)
{
	Camera.OnMouseWheel(zDelta);
}

void COpenGLView::OnPaint()
{
	static DWORD LastFPSTime = GetTickCount(), LastFrameTime = LastFPSTime, FPS = 0;

	PAINTSTRUCT ps;

	HDC hDC = BeginPaint(hWnd, &ps);

	DWORD Time = GetTickCount();

	float FrameTime = (Time - LastFrameTime) * 0.001f;

	LastFrameTime = Time;

	if(Time - LastFPSTime > 1000)
	{
		CString Text = Title;

		if(OpenGLRenderer.Text[0] != 0)
		{
			Text.Append(" - " + OpenGLRenderer.Text);
		}

		Text.Append(" - %dx%d", Width, Height);
		Text.Append(", ATF %dx", gl_max_texture_max_anisotropy_ext);
		Text.Append(", MSAA %dx", Samples);
		Text.Append(", FPS: %d", FPS);
		Text.Append(" - %s", glGetString(GL_RENDERER));
		
		SetWindowText(hWnd, Text);

		LastFPSTime = Time;
		FPS = 0;
	}
	else
	{
		FPS++;
	}

	BYTE Keys = 0x00;

	if(GetKeyState('W') & 0x80) Keys |= 0x01;
	if(GetKeyState('S') & 0x80) Keys |= 0x02;
	if(GetKeyState('A') & 0x80) Keys |= 0x04;
	if(GetKeyState('D') & 0x80) Keys |= 0x08;
	/*if(GetKeyState('R') & 0x80) Keys |= 0x10;
	if(GetKeyState('F') & 0x80) Keys |= 0x20;*/

	if(GetKeyState(VK_SHIFT) & 0x80) Keys |= 0x40;
	if(GetKeyState(VK_CONTROL) & 0x80) Keys |= 0x80;

	if(Keys & 0x3F)
	{
		Camera.Move(Camera.OnKeys(Keys, FrameTime));

		OpenGLRenderer.CheckCameraTerrainPosition();
	}

	OpenGLRenderer.Render(FrameTime);

	SwapBuffers(hDC);

	EndPaint(hWnd, &ps);

	InvalidateRect(hWnd, NULL, FALSE);
}

void COpenGLView::OnRButtonDown(int X, int Y)
{
	LastX = X;
	LastY = Y;
}

void COpenGLView::OnSize(int Width, int Height)
{
	this->Width = Width;
	this->Height = Height;

	OpenGLRenderer.Resize(Width, Height);
}

// ----------------------------------------------------------------------------------------------------------------------------

COpenGLView OpenGLView;

// ----------------------------------------------------------------------------------------------------------------------------

LRESULT CALLBACK WndProc(HWND hWnd, UINT uiMsg, WPARAM wParam, LPARAM lParam)
{
	switch(uiMsg)
	{
		case WM_CLOSE:
			PostQuitMessage(0);
			break;

		case WM_KEYDOWN:
			OpenGLView.OnKeyDown((UINT)wParam);
			break;

		case WM_LBUTTONDOWN:
			OpenGLView.OnLButtonDown(LOWORD(lParam), HIWORD(lParam));
			break;

		case WM_LBUTTONUP:
			OpenGLView.OnLButtonUp(LOWORD(lParam), HIWORD(lParam));
			break;

		case WM_MOUSEMOVE:
			OpenGLView.OnMouseMove(LOWORD(lParam), HIWORD(lParam));
			break;

		case 0x020A: // WM_MOUSWHEEL
			OpenGLView.OnMouseWheel(HIWORD(wParam));
			break;

		case WM_PAINT:
			OpenGLView.OnPaint();
			break;

		case WM_RBUTTONDOWN:
			OpenGLView.OnRButtonDown(LOWORD(lParam), HIWORD(lParam));
			break;

		case WM_SIZE:
			OpenGLView.OnSize(LOWORD(lParam), HIWORD(lParam));
			break;

		default:
			return DefWindowProc(hWnd, uiMsg, wParam, lParam);
	}

	return 0;
}

// ----------------------------------------------------------------------------------------------------------------------------

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR sCmdLine, int iShow)
{
	char *AppName = "Terrain editor";

	if(OpenGLView.Init(hInstance, AppName, 800, 600, 4))
	{
		OpenGLView.Show(true);
		OpenGLView.MessageLoop();
	}
	else
	{
		MessageBox(NULL, ErrorLog, AppName, MB_OK | MB_ICONERROR);
	}

	OpenGLView.Destroy();

	return 0;
}
