// ----------------------------------------------------------------------------------------------------------------------------
//
// Version 2.04
//
// ----------------------------------------------------------------------------------------------------------------------------

#include <windows.h>

#ifndef WM_MOUSWHEEL
	#define WM_MOUSWHEEL 0x020A
#endif

#include "glmath.h"
#include "string.h"

#include <gl/glew.h> // http://glew.sourceforge.net/
#include <gl/wglew.h>

#include <FreeImage.h> // http://freeimage.sourceforge.net/

// ----------------------------------------------------------------------------------------------------------------------------

#pragma comment(lib, "opengl32.lib")
#pragma comment(lib, "glu32.lib")
#pragma comment(lib, "glew32.lib")
#pragma comment(lib, "FreeImage.lib")

// ----------------------------------------------------------------------------------------------------------------------------

extern CString ModuleDirectory, ErrorLog;

// ----------------------------------------------------------------------------------------------------------------------------

#define BUFFER_SIZE_INCREMENT 1048576

// ----------------------------------------------------------------------------------------------------------------------------

class CBuffer
{
private:
	BYTE *Buffer;
	int BufferSize;

private:
	int Position;

public:
	CBuffer();
	~CBuffer();

private:
	void SetDefaults();

public:
	void AddData(void *Data, int DataSize);
	void Empty();
	void *GetData();
	int GetDataSize();
};

// ----------------------------------------------------------------------------------------------------------------------------

extern int gl_max_texture_size, gl_max_texture_max_anisotropy_ext;

// ----------------------------------------------------------------------------------------------------------------------------

class CTexture
{
private:
	GLuint Texture;

private:
	int Width, Height;

public:
	CTexture();
	~CTexture();

private:
	void SetDefaults();

public:
	operator GLuint ();

private:
	FIBITMAP *GetBitmap(char *FileName, int &Width, int &Height, int &BPP);

public:
	bool LoadTexture2D(char *FileName);
	bool LoadTextureCubeMap(char **FileNames);
	void Destroy();

public:
	int GetWidth();
	int GetHeight();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CShaderProgram
{
private:
	GLuint VertexShader, FragmentShader;

private:
	GLuint Program;

public:
	GLuint *UniformLocations, *AttribLocations;

public:
	CShaderProgram();
	~CShaderProgram();

private:
	void SetDefaults();

public:
	operator GLuint ();

private:
	GLuint LoadShader(char *FileName, GLenum Type);

public:
	bool Load(char *VertexShaderFileName, char *FragmentShaderFileName);
	void Destroy();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CPlane
{
private:
	vec3 N;
	float ND;
	int O;

public:
	CPlane();
	~CPlane();

public:
	void Set(const vec3 &A, const vec3 &B, const vec3 &C);
	bool AABBBehind(const vec3 *AABBVertices);
	float AABBDistance(const vec3 *AABBVertices);
};

// ----------------------------------------------------------------------------------------------------------------------------

class CFrustum
{
private:
	vec3 Vertices[8];

private:
	CPlane Planes[6];

public:
	CFrustum();
	~CFrustum();

public:
	void Set(const mat4x4 &ViewProjectionMatrixInverse);
	bool AABBVisible(const vec3 *AABBVertices);
	float AABBDistance(const vec3 *AABBVertices);
	void Render();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CCamera
{
public:
	vec3 X, Y, Z, Position, Reference;

public:
	mat4x4 ViewMatrix, ViewMatrixInverse, ProjectionMatrix, ProjectionMatrixInverse, ViewProjectionMatrix, ViewProjectionMatrixInverse;

public:
	CFrustum Frustum;

public:
	CCamera();
	~CCamera();

public:
	void Look(const vec3 &Position, const vec3 &Reference, bool RotateAroundReference = false);
	void Move(const vec3 &Movement);
	vec3 OnKeys(BYTE Keys, float FrameTime);
	void OnMouseMove(int dx, int dy);
	void OnMouseWheel(float zDelta);
	void SetPerspective(float fovy, float aspect, float n, float f);

private:
	void CalculateViewMatrix();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CVertex
{
public:
	vec3 Position;
	vec3 Normal;
};

// ----------------------------------------------------------------------------------------------------------------------------

class CAABB
{
private:
	vec3 Vertices[8];

public:
	CAABB();
	~CAABB();

public:
	void Set(const vec3 &Min, const vec3 &Max);
	bool PointInside(const vec3 &Point);
	bool Visible(CFrustum &Frustum);
	float Distance(CFrustum &Frustum);
	void Render();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CBSPTreeNode
{
private:
	vec3 Min, Max;

private:
	int Depth;

private:
	CAABB AABB;

private:
	bool Visible;
	float Distance;

private:
	int *Indices;

private:
	int IndicesCount;

private:
	GLuint IndexBufferObject;

private:
	CBSPTreeNode *Children[2];

public:
	CBSPTreeNode();
	~CBSPTreeNode();

private:
	void SetDefaults();

public:
	void InitAABB(const vec3 &Min, const vec3 &Max, int Depth, float MinAABBSize);
	bool CheckTriangle(CVertex *Vertices, int *Indices, int A, int B, int C);
	void AllocateMemory();
	bool AddTriangle(CVertex *Vertices, int *Indices, int A, int B, int C);
	void ResetAABB(CVertex *Vertices);
	int InitIndexBufferObject();
	int CheckVisibility(CFrustum &Frustum, CBSPTreeNode **VisibleGeometryNodes, int &VisibleGeometryNodesCount);
	float GetDistance();
	void Render();
	void RenderAABB(int Depth);
	void Destroy();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CBSPTree
{
private:
	CBSPTreeNode *Root;

private:
	CBSPTreeNode **VisibleGeometryNodes;
	int VisibleGeometryNodesCount;

public:
	CBSPTree();
	~CBSPTree();

private:
	void SetDefaults();

public:
	void Init(CVertex *Vertices, int *Indices, int IndicesCount, const vec3 &Min, const vec3 &Max, float MinAABBSize = 16.0f);
	void QuickSortVisibleGeometryNodes(int Left, int Right);
	int CheckVisibility(CFrustum &Frustum, bool SortVisibleGeometryNodes);
	void Render(bool VisualizeRenderingOrder);
	void RenderAABB(int Depth);
	void Destroy();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CTerrain
{
private:
	int Size, SizeP1;
	float SizeD2;

private:
	vec3 Min, Max;

private:
	float *Heights;

private:
	int VerticesCount, IndicesCount;

private:
	GLuint VertexBufferObject, IndexBufferObject;

public:
	CBSPTree BSPTree;

public:
	CTerrain();
	~CTerrain();

private:
	void SetDefaults();

public:
	bool LoadTexture2D(char *FileName, float Scale = 256.0f, float Offset = -128.0f);
	bool LoadBinary(char *FileName);
	bool SaveBinary(char *FileName);
	int CheckVisibility(CFrustum &Frustum, bool SortVisibleGeometryNodes = true);
	void Render(bool VisualizeRenderingOrder = false);
	void RenderSlow();
	void RenderSlowToShadowMap();
	void RenderAABB(int Depth = -1);
	void Destroy();

public:
	int GetSize();
	vec3 GetMin();
	vec3 GetMax();
	void GetMinMax(mat4x4 &ViewMatrix, vec3 &Min, vec3 &Max);
	int GetTrianglesCount();

private:
	int GetIndex(int X, int Z);
	float GetHeight(int X, int Z);

public:
	float GetHeight(float X, float Z);

private:
	float GetHeight(float *Heights, int Size, float X, float Z);
};

// ----------------------------------------------------------------------------------------------------------------------------

#define SHADOW_MAP_SIZE 4096

// ----------------------------------------------------------------------------------------------------------------------------

class COpenGLRenderer
{
private:
	int LastX, LastY, LastClickedX, LastClickedY;

private:
	int Width, Height;

private:
	CCamera Camera;

private:
	CShaderProgram Shader;

private:
	CTerrain Terrain;

private:
	float LightAngle;

private:
	mat4x4 LightViewMatrix, LightProjectionMatrix, ShadowMatrix;

private:
	int ShadowMapSize;
	GLuint ShadowMap, RotationTexture, FBO;

private:
	bool Wireframe, DisplayShadowMap, RenderAABB, VisualizeRenderingOrder, SortVisibleGeometryNodes, RenderSlow;
	int Depth;

public:
	COpenGLRenderer();
	~COpenGLRenderer();

public:
	bool Init();
	void Render();
	void Animate(float FrameTime);
	void Resize(int Width, int Height);
	void Destroy();

private:
	void RenderShadowMap();
	void CheckCameraTerrainPosition(vec3 &Movement);

public:
	void CheckCameraKeys(float FrameTime);

public:
	void OnKeyDown(UINT Key);
	void OnLButtonDown(int X, int Y);
	void OnLButtonUp(int X, int Y);
	void OnMouseMove(int X, int Y);
	void OnMouseWheel(short zDelta);
	void OnRButtonDown(int X, int Y);
	void OnRButtonUp(int X, int Y);
};

// ----------------------------------------------------------------------------------------------------------------------------

class COpenGLView
{
private:
	HWND hWnd;
	HGLRC hGLRC;

private:
	int Width, Height, Samples;

private:
	CString Title, Resolution, MSAA, ATF, FPS, Renderer;

private:
	COpenGLRenderer OpenGLRenderer;

public:
	COpenGLView();
	~COpenGLView();

public:
	bool Init(HINSTANCE hInstance, char *Title, int Width, int Height, int Samples);
	void Show(bool Maximized = false);
	void MessageLoop();
	void Destroy();

public:
	void OnKeyDown(UINT Key);
	void OnLButtonDown(int X, int Y);
	void OnLButtonUp(int X, int Y);
	void OnMouseMove(int X, int Y);
	void OnMouseWheel(short zDelta);
	void OnPaint();
	void OnRButtonDown(int X, int Y);
	void OnRButtonUp(int X, int Y);
	void OnSize(int Width, int Height);
};

// ----------------------------------------------------------------------------------------------------------------------------

LRESULT CALLBACK WndProc(HWND hWnd, UINT uiMsg, WPARAM wParam, LPARAM lParam);

// ----------------------------------------------------------------------------------------------------------------------------

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR sCmdLine, int iShow);
