// ----------------------------------------------------------------------------------------------------------------------------

#include <windows.h>

#ifndef WM_MOUSWHEEL
	#define WM_MOUSWHEEL 0x020A
#endif

#include "string.h"
#include "glmath.h"

#include <FreeImage.h> // http://freeimage.sourceforge.net/

#pragma comment(lib, "FreeImage.lib")
#pragma comment(lib, "winmm.lib")

#pragma warning(disable : 4996)

// ----------------------------------------------------------------------------------------------------------------------------

class CTexture
{
private:
	void *Bits;

private:
	int Width, Height, Format, Pitch;

public:
	CTexture();
	~CTexture();

private:
	void SetDefaults();

public:
	void* GetBits();
	int GetWidth();
	int GetHeight();
	int GetFormat();

public:
	void Create(int Width, int Height, int Format);
	bool LoadTexture(char *TextureFileName);
	void GetColorNearest(float s, float t, vec3 &Color);
	void GetColorBilinear(float s, float t, vec3 &Color);
	float GetShadowNearest(vec4 &TexCoord);
	float GetShadowBilinear(vec4 &TexCoord);
	void Destroy();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CFrameBuffer
{
private:
	CTexture *ColorTexture;
	CTexture *DepthTexture;

public:
	CFrameBuffer();
	~CFrameBuffer();

public:
	CTexture* GetColorTexture();
	void SetColorTexture(CTexture *ColorTexture);
	CTexture* GetDepthTexture();
	void SetDepthTexture(CTexture *DepthTexture);
};

// ----------------------------------------------------------------------------------------------------------------------------

class CCamera
{
public:
	vec3 X, Y, Z, Position, Reference;

public:
	mat4x4 ViewMatrix, ProjectionMatrix, ViewProjectionMatrix;

public:
	CCamera();
	~CCamera();

public:
	void Look(const vec3 &Position, const vec3 &Reference, bool RotateAroundReference = false);
	void Move(const vec3 &Movement);
	vec3 OnKeys(BYTE Keys, float FrameTime);
	void OnMouseMove(int dx, int dy);
	void OnMouseWheel(float zDelta);
	void SetPerspective(float fovy, float aspect, float n, float f);

private:
	void CalculateViewMatrix();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CVertex
{
public:
	vec3 Position;
	vec3 Color;
	vec2 TexCoord;
	vec3 Normal;
};

// ----------------------------------------------------------------------------------------------------------------------------

class CObject
{
public:
	CTexture Texture;

public:
	CVertex *Vertices;
	int VerticesCount;

public:
	vec3 Min, Max;

public:
	CObject();
	~CObject();

private:
	void SetDefaults();

public:
	bool Load(char *Directory, char *ObjFileName);
	void Translate(const vec3 &Translation);
	void Scale(float ScaleFactor);
	void Rotate(float Angle, const vec3 &Axis);
	void Destroy();

private:
	bool ReadSource(char *Directory, char *FileName, char **Source, long &Length);
	bool ParseMtl(char *Directory, char *MtlFileName);
	void GetMinMax();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CLight
{
public:
	vec3 Position;
	vec3 Ambient;
	vec3 Diffuse;
	float ConstantAttenuation;
	float LinearAttenuation;
	float QuadraticAttenuation;
};

// ----------------------------------------------------------------------------------------------------------------------------

class CFragment
{
public:
	vec4 Position;
	vec3 Color;
	vec2 TexCoord;
	vec3 Normal;
	vec3 LightDirection;
	vec4 ShadowMapTexCoord;
};

// ----------------------------------------------------------------------------------------------------------------------------

class CEdge
{
public:
	CFragment *Fragment1, *Fragment2;

public:
	vec4 PositionDiff;
	vec3 ColorDiff;
	vec2 TexCoordDiff;
	vec3 NormalDiff;
	vec3 LightDirectionDiff;
	vec4 ShadowMapTexCoordDiff;

public:
	void Set(CFragment *Fragment1, CFragment *Fragment2);
};

// ----------------------------------------------------------------------------------------------------------------------------

class CSpan
{
public:
	CFragment *Fragment1, *Fragment2;

public:
	vec4 PositionDiff;
	vec3 ColorDiff;
	vec2 TexCoordDiff;
	vec3 NormalDiff;
	vec3 LightDirectionDiff;
	vec4 ShadowMapTexCoordDiff;

public:
	void Set(CFragment *Fragment1, CFragment *Fragment2);
};

// ----------------------------------------------------------------------------------------------------------------------------

#define NONE 0x00

#define ANTI_ALIASING_2X2 0x01
#define ANTI_ALIASING_3X3 0x02
#define ANTI_ALIASING_4X4 0x03

#define CULL_FACE_FRONT 0x04
#define CULL_FACE_BACK 0x05

#define TEXTURE_FORMAT_BGR24 0x06
#define TEXTURE_FORMAT_DEPTH16 0x07

// ----------------------------------------------------------------------------------------------------------------------------

#define THREAD_SLEEP_TIME 0

// ----------------------------------------------------------------------------------------------------------------------------

class CThreadData
{
public:
	HANDLE Thread;
	DWORD Id;
	void *SoftwareGL;
	int ThreadId, Function;
	bool ThreadIsRunning;
};

// ----------------------------------------------------------------------------------------------------------------------------

class CSoftwareGL
{
private:
	int Width, Height;

private:
	int ViewportX, ViewportY, ViewportWidth, ViewportHeight;

private:
	BYTE *AntiAliasingColorBuffer;
	int AntiAliasingColorBufferWidth, AntiAliasingColorBufferHeight;

private:
	BYTE *StandardColorBuffer;
	int StandardColorBufferWidth, StandardColorBufferHeight;

private:
	BITMAPINFO BitmapInfo;

private:
	USHORT *StandardDepthBuffer;
	int StandardDepthBufferWidth, StandardDepthBufferHeight;

private:
	BYTE *ColorBuffer;
	int ColorBufferWidth, ColorBufferHeight;

private:
	USHORT *DepthBuffer;
	int DepthBufferWidth, DepthBufferHeight;

private:
	int BufferWidthM1, BufferHeightM1;

private:
	CFrameBuffer *FrameBuffer;

private:
	int AntiAliasing;
	mat4x4 ModelViewProjectionMatrix, ShadowMapMatrix;
	CLight *Light;
	int CullFace;
	bool BilinearTextureFiltering, DepthTest;
	CTexture *Texture, *ShadowMap;

private:
	CThreadData *ThreadsData;
	int ThreadsCount;

private:
	bool ColorBuffering, DepthTesting, NotDepthTesting, Texturing, Lighting, ShadowMapping;

private:
	CVertex *Vertices;
	int FirstIndex, LastIndex;

public:
	CSoftwareGL();
	~CSoftwareGL();

public:
	int GetAntiAliasing();
	void SetAntiAliasing(int AntiAliasing);
	int GetCullFace();
	void SetCullFace(int CullFace);
	bool GetBilinearTextureFiltering();
	void SetBilinearTextureFiltering(bool BilinearTextureFiltering);
	bool GetDepthTest();
	void SetDepthTest(bool DepthTest);

public:
	int GetThreadsCount();

public:
	void BindFrameBuffer(CFrameBuffer *FrameBuffer);
	void Viewport(int X, int Y, int Width, int Height);
	void Clear(bool ClearColorBuffer = true, bool ClearDepthBuffer = true);
	void LoadModelViewProjectionMatrix(const mat4x4 &ModelViewProjectionMatrix);
	void LoadShadowMapMatrix(const mat4x4 &ShadowMapMatrix);
	void BindLight(CLight *Light);
	void BindTexture(CTexture *Texture);
	void BindShadowMap(CTexture *ShadowMap);
	void DrawTriangles(CVertex *Vertices, int FirstIndex, int Count);

private:
	void RunFunctionMultiThreadedAndWaitForCompletion(int Function);
	bool ThreadsAreRunning();
	static DWORD WINAPI ThreadProc(LPVOID lpParam);
	void DrawTriangles(int ThreadId);
	void ClipTriangle(CFragment *Fragments, int ThreadId, int ClipPlane = 1);
	void RasterizeTriangle(CFragment *Fragments, int ThreadId);
	void DrawSpansBetweenEdges(CEdge *Edge1, CEdge *Edge2, int ThreadId);
	void DrawSpan(CSpan *Span, int y);
	void BlitAntiAliasingColorBuffer2x2(int Line);
	void BlitAntiAliasingColorBuffer3x3(int Line);
	void BlitAntiAliasingColorBuffer4x4(int Line);

public:
	void ResizeBuffers(int Width, int Height);
	void SwapBuffers(HDC hDC);
};

// ----------------------------------------------------------------------------------------------------------------------------

class CSoftwareGLRenderer : public CSoftwareGL
{
private:
	int Width, Height;

private:
	int LastX, LastY, LastClickedX, LastClickedY;

private:
	CCamera Camera;

private:
	CTexture Texture;

private:
	CObject Object;

private:
	CVertex *Vertices;

private:
	CLight Light;
	mat4x4 LightViewMatrix, LightProjectionMatrix, LightViewProjectionMatrix;

private:
	CTexture ShadowMap;
	CFrameBuffer FrameBuffer;
	mat4x4 OrthogonalProjectionMatrix;

private:
	bool RenderObject, Texturing, Lighting, ShadowMapping, DisplayShadowMap, RotateLight;

public:
	CSoftwareGLRenderer();
	~CSoftwareGLRenderer();

public:
	bool Init();
	void Animate(float FrameTime);
	void Render();
	void Resize(int Width, int Height);
	void Destroy();

private:
	void RenderShadowMap();

public:
	void CheckCameraKeys(float FrameTime);

public:
	void OnKeyDown(UINT Key);
	void OnLButtonDown(int X, int Y);
	void OnLButtonUp(int X, int Y);
	void OnMouseMove(int X, int Y);
	void OnMouseWheel(short zDelta);
	void OnRButtonDown(int X, int Y);
	void OnRButtonUp(int X, int Y);
};

// ----------------------------------------------------------------------------------------------------------------------------

class CSoftwareGLView
{
private:
	char *Title;
	int Width, Height;
	HWND hWnd;
	HDC hDC;

private:
	CSoftwareGLRenderer SoftwareGLRenderer;

public:
	CSoftwareGLView();
	~CSoftwareGLView();

public:
	bool Create(HINSTANCE hInstance, char *Title, int Width, int Height);
	void Show(bool Maximized = false);
	void MsgLoop();
	void Destroy();

public:
	void OnKeyDown(UINT Key);
	void OnLButtonDown(int X, int Y);
	void OnLButtonUp(int X, int Y);
	void OnMouseMove(int X, int Y);
	void OnMouseWheel(short zDelta);
	void OnPaint();
	void OnRButtonDown(int X, int Y);
	void OnRButtonUp(int X, int Y);
	void OnSize(int Width, int Height);
};

// ----------------------------------------------------------------------------------------------------------------------------

LRESULT CALLBACK WndProc(HWND hWnd, UINT uiMsg, WPARAM wParam, LPARAM lParam);
int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR sCmdLine, int iShow);

// ----------------------------------------------------------------------------------------------------------------------------
