#include "opengl_21_tutorials_win32_framework.h"

// ----------------------------------------------------------------------------------------------------------------------------

CBuffer::CBuffer()
{
	SetDefaults();
}

CBuffer::~CBuffer()
{
	Empty();
}

void CBuffer::AddData(void *Data, int DataSize)
{
	int Remaining = BufferSize - Position;

	if(DataSize > Remaining)
	{
		BYTE *OldBuffer = Buffer;
		int OldBufferSize = BufferSize;

		int Needed = DataSize - Remaining;

		BufferSize += Needed > BUFFER_SIZE_INCREMENT ? Needed : BUFFER_SIZE_INCREMENT;

		Buffer = new BYTE[BufferSize];

		memcpy(Buffer, OldBuffer, OldBufferSize);

		delete [] OldBuffer;
	}

	memcpy(Buffer + Position, Data, DataSize);

	Position += DataSize;
}

void CBuffer::Empty()
{
	delete [] Buffer;

	SetDefaults();
}

void *CBuffer::GetData()
{
	return Buffer;
}

int CBuffer::GetDataSize()
{
	return Position;
}

void CBuffer::SetDefaults()
{
	Buffer = NULL;

	BufferSize = 0;
	Position = 0;
}

// ----------------------------------------------------------------------------------------------------------------------------

int gl_max_texture_size = 0, gl_max_texture_max_anisotropy_ext = 0;

// ----------------------------------------------------------------------------------------------------------------------------

CTexture::CTexture()
{
	Texture = 0;
}

CTexture::~CTexture()
{
}

CTexture::operator GLuint ()
{
	return Texture;
}

bool CTexture::LoadTexture2D(char *FileName)
{
	CString DirectoryFileName = ModuleDirectory + "Textures\\" + FileName;

	int Width, Height, BPP;

	FIBITMAP *dib = GetBitmap(DirectoryFileName, Width, Height, BPP);

	if(dib == NULL)
	{
		ErrorLog.Append("Error loading texture " + DirectoryFileName + "!\r\n");
		return false;
	}

	GLenum Format = 0;

	if(BPP == 32) Format = GL_BGRA;
	if(BPP == 24) Format = GL_BGR;

	if(Format == 0)
	{
		ErrorLog.Append("Unsupported texture format (%s)!\r\n", FileName);
		FreeImage_Unload(dib);
		return false;
	}

	Destroy();

	glGenTextures(1, &Texture);

	glBindTexture(GL_TEXTURE_2D, Texture);

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	if(GLEW_EXT_texture_filter_anisotropic)
	{
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY_EXT, gl_max_texture_max_anisotropy_ext);
	}

	glTexParameteri(GL_TEXTURE_2D, GL_GENERATE_MIPMAP, GL_TRUE);

	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, Width, Height, 0, Format, GL_UNSIGNED_BYTE, FreeImage_GetBits(dib));

	glBindTexture(GL_TEXTURE_2D, 0);

	FreeImage_Unload(dib);

	return true;
}

bool CTexture::LoadTextureCubeMap(char **FileNames)
{
	int Width, Height, BPP;

	FIBITMAP *dib[6];

	bool Error = false;

	for(int i = 0; i < 6; i++)
	{
		CString DirectoryFileName = ModuleDirectory + "Textures\\" + FileNames[i];

		dib[i] = GetBitmap(DirectoryFileName, Width, Height, BPP);

		if(dib[i] == NULL)
		{
			ErrorLog.Append("Error loading texture " + DirectoryFileName + "!\r\n");
			Error = true;
		}
	}

	if(Error)
	{
		for(int i = 0; i < 6; i++)
		{
			FreeImage_Unload(dib[i]);
		}

		return false;
	}

	GLenum Format = 0;

	if(BPP == 32) Format = GL_BGRA;
	if(BPP == 24) Format = GL_BGR;

	if(Format == 0)
	{
		ErrorLog.Append("Unsupported texture format (%s)!\r\n", FileNames[5]);

		for(int i = 0; i < 6; i++)
		{
			FreeImage_Unload(dib[i]);
		}

		return false;
	}

	Destroy();

	glGenTextures(1, &Texture);

	glBindTexture(GL_TEXTURE_CUBE_MAP, Texture);

	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	if(GLEW_EXT_texture_filter_anisotropic)
	{
		glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MAX_ANISOTROPY_EXT, gl_max_texture_max_anisotropy_ext);
	}

	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_GENERATE_MIPMAP, GL_TRUE);

	for(int i = 0; i < 6; i++)
	{
		glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_X + i, 0, GL_RGBA8, Width, Height, 0, Format, GL_UNSIGNED_BYTE, FreeImage_GetBits(dib[i]));
	}

	glBindTexture(GL_TEXTURE_CUBE_MAP, 0);

	for(int i = 0; i < 6; i++)
	{
		FreeImage_Unload(dib[i]);
	}

	return true;
}

void CTexture::Destroy()
{
	glDeleteTextures(1, &Texture);
	Texture = 0;
}

FIBITMAP *CTexture::GetBitmap(char *FileName, int &Width, int &Height, int &BPP)
{
	FREE_IMAGE_FORMAT fif = FreeImage_GetFileType(FileName);

	if(fif == FIF_UNKNOWN)
	{
		fif = FreeImage_GetFIFFromFilename(FileName);
	}

	if(fif == FIF_UNKNOWN)
	{
		return NULL;
	}

	FIBITMAP *dib = NULL;

	if(FreeImage_FIFSupportsReading(fif))
	{
		dib = FreeImage_Load(fif, FileName);
	}

	if(dib != NULL)
	{
		int OriginalWidth = FreeImage_GetWidth(dib);
		int OriginalHeight = FreeImage_GetHeight(dib);

		Width = OriginalWidth;
		Height = OriginalHeight;

		if(Width == 0 || Height == 0)
		{
			FreeImage_Unload(dib);
			return NULL;
		}

		BPP = FreeImage_GetBPP(dib);

		if(Width > gl_max_texture_size) Width = gl_max_texture_size;
		if(Height > gl_max_texture_size) Height = gl_max_texture_size;

		if(!GLEW_ARB_texture_non_power_of_two)
		{
			Width = 1 << (int)floor((log((float)Width) / log(2.0f)) + 0.5f); 
			Height = 1 << (int)floor((log((float)Height) / log(2.0f)) + 0.5f);
		}

		if(Width != OriginalWidth || Height != OriginalHeight)
		{
			FIBITMAP *rdib = FreeImage_Rescale(dib, Width, Height, FILTER_BICUBIC);
			FreeImage_Unload(dib);
			dib = rdib;
		}
	}

	return dib;
}

// ----------------------------------------------------------------------------------------------------------------------------

CShaderProgram::CShaderProgram()
{
	SetDefaults();
}

CShaderProgram::~CShaderProgram()
{
}

CShaderProgram::operator GLuint ()
{
	return Program;
}

bool CShaderProgram::Load(char *VertexShaderFileName, char *FragmentShaderFileName)
{
	bool Error = false;

	Destroy();

	Error |= ((VertexShader = LoadShader(VertexShaderFileName, GL_VERTEX_SHADER)) == 0);
	Error |= ((FragmentShader = LoadShader(FragmentShaderFileName, GL_FRAGMENT_SHADER)) == 0);

	if(Error)
	{
		Destroy();
		return false;
	}

	Program = glCreateProgram();
	glAttachShader(Program, VertexShader);
	glAttachShader(Program, FragmentShader);
	glLinkProgram(Program);

	int LinkStatus;
	glGetProgramiv(Program, GL_LINK_STATUS, &LinkStatus);

	if(LinkStatus == GL_FALSE)
	{
		ErrorLog.Append("Error linking program (%s, %s)!\r\n", VertexShaderFileName, FragmentShaderFileName);

		int InfoLogLength = 0;
		glGetProgramiv(Program, GL_INFO_LOG_LENGTH, &InfoLogLength);

		if(InfoLogLength > 0)
		{
			char *InfoLog = new char[InfoLogLength];
			int CharsWritten  = 0;
			glGetProgramInfoLog(Program, InfoLogLength, &CharsWritten, InfoLog);
			ErrorLog.Append(InfoLog);
			delete [] InfoLog;
		}

		Destroy();

		return false;
	}

	return true;
}

void CShaderProgram::Destroy()
{
	glDetachShader(Program, VertexShader);
	glDetachShader(Program, FragmentShader);

	glDeleteShader(VertexShader);
	glDeleteShader(FragmentShader);

	glDeleteProgram(Program);

	delete [] UniformLocations;
	delete [] AttribLocations;

	SetDefaults();
}

GLuint CShaderProgram::LoadShader(char *FileName, GLenum Type)
{
	CString DirectoryFileName = ModuleDirectory + "Shaders\\" + FileName;

	FILE *File;

	if(fopen_s(&File, DirectoryFileName, "rb") != 0)
	{
		ErrorLog.Append("Error loading file " + DirectoryFileName + "!\r\n");
		return 0;
	}

	fseek(File, 0, SEEK_END);
	long Size = ftell(File);
	fseek(File, 0, SEEK_SET);
	char *Source = new char[Size + 1];
	fread(Source, 1, Size, File);
	fclose(File);
	Source[Size] = 0;

	GLuint Shader = glCreateShader(Type);

	glShaderSource(Shader, 1, (const char**)&Source, NULL);
	delete [] Source;
	glCompileShader(Shader);

	int CompileStatus;
	glGetShaderiv(Shader, GL_COMPILE_STATUS, &CompileStatus);

	if(CompileStatus == GL_FALSE)
	{
		ErrorLog.Append("Error compiling shader %s!\r\n", FileName);

		int InfoLogLength = 0;
		glGetShaderiv(Shader, GL_INFO_LOG_LENGTH, &InfoLogLength);

		if(InfoLogLength > 0)
		{
			char *InfoLog = new char[InfoLogLength];
			int CharsWritten  = 0;
			glGetShaderInfoLog(Shader, InfoLogLength, &CharsWritten, InfoLog);
			ErrorLog.Append(InfoLog);
			delete [] InfoLog;
		}

		glDeleteShader(Shader);

		return 0;
	}

	return Shader;
}

void CShaderProgram::SetDefaults()
{
	VertexShader = 0;
	FragmentShader = 0;

	Program = 0;

	UniformLocations = NULL;
	AttribLocations = NULL;
}

// ----------------------------------------------------------------------------------------------------------------------------

CCamera::CCamera()
{
	X = vec3(1.0f, 0.0f, 0.0f);
	Y = vec3(0.0f, 1.0f, 0.0f);
	Z = vec3(0.0f, 0.0f, 1.0f);

	Position = vec3(0.0f, 0.0f, 5.0f);
	Reference = vec3(0.0f, 0.0f, 0.0f);

	CalculateViewMatrix();
}

CCamera::~CCamera()
{
}

void CCamera::Look(const vec3 &Position, const vec3 &Reference, bool RotateAroundReference)
{
	this->Position = Position;
	this->Reference = Reference;

	Z = normalize(Position - Reference);
	X = normalize(cross(vec3(0.0f, 1.0f, 0.0f), Z));
	Y = cross(Z, X);

	if(!RotateAroundReference)
	{
		this->Reference = this->Position - Z * 0.05f;
	}

	CalculateViewMatrix();
}

void CCamera::Move(const vec3 &Movement)
{
	Position += Movement;
	Reference += Movement;

	CalculateViewMatrix();
}

vec3 CCamera::OnKeys(BYTE Keys, float FrameTime)
{
	float Speed = 5.0f;

	if(Keys & 0x40) Speed *= 2.0f;
	if(Keys & 0x80) Speed *= 0.5f;

	float Distance = Speed * FrameTime;

	vec3 Up(0.0f, 1.0f, 0.0f);
	vec3 Right = X;
	vec3 Forward = cross(Up, Right);

	Up *= Distance;
	Right *= Distance;
	Forward *= Distance;

	vec3 Movement;

	if(Keys & 0x01) Movement += Forward;
	if(Keys & 0x02) Movement -= Forward;
	if(Keys & 0x04) Movement -= Right;
	if(Keys & 0x08) Movement += Right;
	if(Keys & 0x10) Movement += Up;
	if(Keys & 0x20) Movement -= Up;

	return Movement;
}

void CCamera::OnMouseMove(int dx, int dy)
{
	float Sensitivity = 0.25f;

	Position -= Reference;

	if(dx != 0)
	{
		float DeltaX = (float)dx * Sensitivity;

		X = rotate(X, DeltaX, vec3(0.0f, 1.0f, 0.0f));
		Y = rotate(Y, DeltaX, vec3(0.0f, 1.0f, 0.0f));
		Z = rotate(Z, DeltaX, vec3(0.0f, 1.0f, 0.0f));
	}

	if(dy != 0)
	{
		float DeltaY = (float)dy * Sensitivity;

		Y = rotate(Y, DeltaY, X);
		Z = rotate(Z, DeltaY, X);

		if(Y.y < 0.0f)
		{
			Z = vec3(0.0f, Z.y > 0.0f ? 1.0f : -1.0f, 0.0f);
			Y = cross(Z, X);
		}
	}

	Position = Reference + Z * length(Position);

	CalculateViewMatrix();
}

void CCamera::OnMouseWheel(float zDelta)
{
	Position -= Reference;

	if(zDelta < 0 && length(Position) < 500.0f)
	{
		Position += Position * 0.1f;
	}

	if(zDelta > 0 && length(Position) > 0.05f)
	{
		Position -= Position * 0.1f;
	}

	Position += Reference;

	CalculateViewMatrix();
}

void CCamera::SetPerspective(float fovy, float aspect, float n, float f)
{
	ProjectionMatrix = perspective(fovy, aspect, n, f);
	ProjectionMatrixInverse = inverse(ProjectionMatrix);
	ViewProjectionMatrix = ProjectionMatrix * ViewMatrix;
	ViewProjectionMatrixInverse = ViewMatrixInverse * ProjectionMatrixInverse;
}

void CCamera::CalculateViewMatrix()
{
	ViewMatrix = mat4x4(X.x, Y.x, Z.x, 0.0f, X.y, Y.y, Z.y, 0.0f, X.z, Y.z, Z.z, 0.0f, -dot(X, Position), -dot(Y, Position), -dot(Z, Position), 1.0f);
	ViewMatrixInverse = inverse(ViewMatrix);
	ViewProjectionMatrix = ProjectionMatrix * ViewMatrix;
	ViewProjectionMatrixInverse = ViewMatrixInverse * ProjectionMatrixInverse;
}

// ----------------------------------------------------------------------------------------------------------------------------

CTriangle::CTriangle()
{
}

CTriangle::CTriangle(const vec3 &A, const vec3 &B, const vec3 &C)
{
	Set(A, B, C);
}

CTriangle::~CTriangle()
{
}

void CTriangle::Set(const vec3 &A, const vec3 &B, const vec3 &C)
{
	this->A = A;
	this->B = B;
	this->C = C;

	M = (A + B + C) / 3.0f;

	AB = B - A;
	BC = C - B;
	CA = A - C;

	LAB = length(AB);
	LBC = length(BC);
	LCA = length(CA);

	AB /= LAB;
	BC /= LBC;
	CA /= LCA;

	N = normalize(cross(AB, -CA));
	D = -dot(N, A);

	NH = (N.y > -1.0f && N.y < 1.0f) ? normalize(vec3(N.x, 0.0f, N.z)) : vec3(0.0f);
	NdotNH = dot(N, NH);

	N1 = normalize(cross(N, AB));
	D1 = -dot(N1, A);

	N2 = normalize(cross(N, BC));
	D2 = -dot(N2, B);

	N3 = normalize(cross(N, CA));
	D3 = -dot(N3, C);

	HPNAB = (AB.y > -1.0f && AB.y < 1.0f) ? normalize(vec3(0.0f, 1.0f, 0.0f) - AB * AB.y) : vec3(0.0f);
	HPDAB = -dot(A, HPNAB);
	VPNAB = cross(AB, HPNAB);
	VPDAB = -dot(A, VPNAB);

	HPNBC = (BC.y > -1.0f && BC.y < 1.0f) ? normalize(vec3(0.0f, 1.0f, 0.0f) - BC * BC.y) : vec3(0.0f);
	HPDBC = -dot(B, HPNBC);
	VPNBC = cross(BC, HPNBC);
	VPDBC = -dot(B, VPNBC);

	HPNCA = (CA.y > -1.0f && CA.y < 1.0f) ? normalize(vec3(0.0f, 1.0f, 0.0f) - CA * CA.y) : vec3(0.0f);
	HPDCA = -dot(C, HPNCA);
	VPNCA = cross(CA, HPNCA);
	VPDCA = -dot(C, VPNCA);
}

bool CTriangle::Inside(const vec3 &Point)
{
	if(dot(N1, Point) + D1 < 0.0f) return false;
	if(dot(N2, Point) + D2 < 0.0f) return false;
	if(dot(N3, Point) + D3 < 0.0f) return false;

	return true;
}

bool CTriangle::RayTriangleIntersectionTest(const vec3 &RayOrigin, const vec3 &RayDirection, float &MinDistance, vec3 &IntersectionPoint)
{
	float NdotRD = -dot(N, RayDirection);

	if(NdotRD > 0.0f)
	{
		float DistanceFromPlane = (dot(N, RayOrigin) + D) / NdotRD;

		if(DistanceFromPlane > 0.0f && DistanceFromPlane < MinDistance)
		{
			vec3 PointOnPlane = RayOrigin + RayDirection * DistanceFromPlane;

			if(Inside(PointOnPlane))
			{
				MinDistance = DistanceFromPlane;
				IntersectionPoint = PointOnPlane;

				return true;
			}
		}
	}

	return false;
}

bool CTriangle::GetHeightAbove(const vec3 &EyePosition, float &MinDistance, float &Height)
{
	float NdotRD = -N.y;

	if(NdotRD > 0.0f)
	{
		float DistanceFromPlane = (dot(N, EyePosition) + D) / NdotRD;

		if(DistanceFromPlane > 0.0f && DistanceFromPlane < MinDistance)
		{
			vec3 PointOnPlane = vec3(EyePosition.x, EyePosition.y + DistanceFromPlane, EyePosition.z);

			if(Inside(PointOnPlane))
			{
				MinDistance = DistanceFromPlane;
				Height = PointOnPlane.y;

				return true;
			}
		}
	}

	return false;
}

bool CTriangle::GetHeightUnder(const vec3 &EyePosition, float EyeKneeDistance, float &MinDistance, float &Height)
{
	float NdotRD = N.y;

	if(NdotRD > 0.0f)
	{
		float DistanceFromPlane = (dot(N, EyePosition) + D) / NdotRD;

		if(DistanceFromPlane > EyeKneeDistance && DistanceFromPlane < MinDistance)
		{
			vec3 PointOnPlane = vec3(EyePosition.x, EyePosition.y - DistanceFromPlane, EyePosition.z);

			if(Inside(PointOnPlane))
			{
				MinDistance = DistanceFromPlane;
				Height = PointOnPlane.y;

				return true;
			}
		}
	}

	return false;
}

bool CTriangle::IntersectionTest(const vec3 &EyePositionA, const vec3 &EyePositionB, const vec3 &Direction, float EyeKneeDistance, float ClosestDistance, const vec3 &PN, float PD, float &MinDistance, vec3 &Compensation)
{
	bool IntersectionTestPassed = false;

	if(NdotNH > 0.0f)
	{
		float NdotD = -dot(N, Direction);

		if(NdotD > 0.0f)
		{
			float DistanceFromPlane = (dot(N, EyePositionA) + D) / NdotD;

			if(DistanceFromPlane > 0.0f && DistanceFromPlane < MinDistance)
			{
				vec3 PointOnPlane = EyePositionA + Direction * DistanceFromPlane;

				if(Inside(PointOnPlane))
				{
					IntersectionTestPassed = true;
					MinDistance = DistanceFromPlane;
					Compensation = PointOnPlane - EyePositionB + NH * (ClosestDistance / NdotNH);
				}
			}
		}
	}

	vec3 *Vertices = (vec3*)&A;
	vec3 *Edges = (vec3*)&AB;
	float *EdgesLengths = &LAB;
	vec3 *VPNs = (vec3*)&VPNAB;

	for(int i = 0; i < 3; i++)
	{
		float PNdotE = -dot(PN, Edges[i]);

		if(PNdotE != 0.0f)
		{
			float DistanceFromPlane = (dot(PN, Vertices[i]) + PD) / PNdotE;

			if(DistanceFromPlane > 0.0f && DistanceFromPlane < EdgesLengths[i])
			{
				vec3 PointOnPlane = Vertices[i] + Edges[i] * DistanceFromPlane;

				vec3 EPAPOP = PointOnPlane - EyePositionA;

				float DistanceV = -EPAPOP.y;

				if(DistanceV > 0.0f && DistanceV < EyeKneeDistance)
				{
					float DistanceH = dot(Direction, EPAPOP);

					if(DistanceH > 0.0f && DistanceH < MinDistance)
					{
						IntersectionTestPassed = true;
						MinDistance = DistanceH;
						Compensation = vec3(PointOnPlane.x - EyePositionB.x, 0.0f, PointOnPlane.z - EyePositionB.z);
						float VPNdotD = -dot(VPNs[i], Direction);
						if(VPNdotD > 0.0f) Compensation += VPNs[i] * ClosestDistance;
						if(VPNdotD < 0.0f) Compensation -= VPNs[i] * ClosestDistance;
					}
				}
			}
		}
	}

	return IntersectionTestPassed;
}

bool CTriangle::DistanceTest(const vec3 &EyePositionB, float EyeKneeDistance, float ClosestDistance, float &MinDistance, vec3 &Compensation)
{
	bool DistanceTestFailed = false;

	if(NdotNH > 0.0f)
	{
		float DistanceFromPlane = dot(N, EyePositionB) + D;

		if(DistanceFromPlane > 0.0f && DistanceFromPlane < MinDistance)
		{
			if(Inside(EyePositionB))
			{
				DistanceTestFailed = true;
				MinDistance = DistanceFromPlane;
				Compensation = NH * ((ClosestDistance - DistanceFromPlane) / NdotNH);
			}
		}
	}

	vec3 *Vertices = (vec3*)&A;
	vec3 *Edges = (vec3*)&AB;
	float *EdgesLengths = &LAB;

	for(int i = 0; i < 3; i++)
	{
		vec3 EPBD = EyePositionB - Vertices[i];

		float EdotEPBD = dot(Edges[i], EPBD);

		if(EdotEPBD > 0.0f && EdotEPBD < EdgesLengths[i])
		{
			vec3 N = EPBD - Edges[i] * EdotEPBD;

			if(N.x != 0.0f || N.z != 0.0f)
			{
				float DistanceFromEdge = length(N);

				if(DistanceFromEdge > 0.0f && DistanceFromEdge < MinDistance)
				{
					DistanceTestFailed = true;
					MinDistance = DistanceFromEdge;
					N /= DistanceFromEdge;
					vec3 NH = normalize(vec3(N.x, 0.0f, N.z));
					float NdotNH = dot(N, NH);
					Compensation = NH * ((ClosestDistance - DistanceFromEdge) / NdotNH);
				}
			}
		}
	}

	for(int i = 0; i < 3; i++)
	{
		vec3 N = EyePositionB - Vertices[i];

		if(N.x != 0.0f || N.z != 0.0f)
		{
			float DistanceFromVertex = length(N);

			if(DistanceFromVertex > 0.0f && DistanceFromVertex < MinDistance)
			{
				DistanceTestFailed = true;
				MinDistance = DistanceFromVertex;
				N /= DistanceFromVertex;
				vec3 NH = normalize(vec3(N.x, 0.0f, N.z));
				float NdotNH = dot(N, NH);
				Compensation = NH * ((ClosestDistance - DistanceFromVertex) / NdotNH);
			}
		}
	}

	vec3 *HPNs = (vec3*)&HPNAB;
	float *HPDs = &HPDAB;
	vec3 *VPNs = (vec3*)&VPNAB;
	float *VPDs = &VPDAB;

	for(int i = 0; i < 3; i++)
	{
		if(HPNs[i].y > 0.0f)
		{
			float DistanceFromHorizontalPlane = (dot(HPNs[i], EyePositionB) + HPDs[i]) / HPNs[i].y;

			if(DistanceFromHorizontalPlane > 0.0f && DistanceFromHorizontalPlane < EyeKneeDistance)
			{
				float DistanceFromVerticalPlane = dot(VPNs[i], EyePositionB) + VPDs[i];

				if(DistanceFromVerticalPlane > 0.0f && DistanceFromVerticalPlane < MinDistance)
				{
					vec3 PointOnHorizontalPlane = vec3(EyePositionB.x, EyePositionB.y - DistanceFromHorizontalPlane, EyePositionB.z);

					float EdotPOHPD = dot(Edges[i], PointOnHorizontalPlane - Vertices[i]);

					if(EdotPOHPD > 0.0f && EdotPOHPD < EdgesLengths[i])
					{
						DistanceTestFailed = true;
						MinDistance = DistanceFromVerticalPlane;
						Compensation = VPNs[i] * (ClosestDistance - DistanceFromVerticalPlane);
					}
				}
			}
		}
	}

	for(int i = 0; i < 3; i++)
	{
		vec3 EPBD = Vertices[i] - EyePositionB;

		float EdotEPBD = -EPBD.y;

		if(EdotEPBD > 0.0f && EdotEPBD < EyeKneeDistance)
		{
			vec3 N = vec3(EPBD.x, EPBD.y + EdotEPBD, EPBD.z);

			float DistanceFromVertex = length(N);

			if(DistanceFromVertex > 0.0f && DistanceFromVertex < MinDistance)
			{
				DistanceTestFailed = true;
				MinDistance = DistanceFromVertex;
				N /= DistanceFromVertex;
				Compensation = N * (DistanceFromVertex - ClosestDistance);
			}
		}
	}

	return DistanceTestFailed;
}

// ----------------------------------------------------------------------------------------------------------------------------

CCollisionDetector::CCollisionDetector()
{
	SetDefaults();
}

CCollisionDetector::~CCollisionDetector()
{
}

void CCollisionDetector::SetDefaults()
{
	Triangles = NULL;
	TrianglesCount = 0;

	EyeHeight = 0.0f;
	EyeKneeDistance = 0.0f;
	ClosestDistance = 0.0f;

	EH = 0.0f;
	EHD2 = 0.0f;
	EKD = 0.0f;
	EKDD2 = 0.0f;

	FallSpeed = 0.0f;
	CrouchState = 0;
}

void CCollisionDetector::Init(vec3 *Vertices, int VerticesCount, float EyeHeight, float EyeKneeDistance, float ClosestDistance)
{
	Destroy();

	this->EyeHeight = EyeHeight;
	this->EyeKneeDistance = EyeKneeDistance;
	this->ClosestDistance = ClosestDistance;

	EH = EyeHeight;
	EHD2 = EyeHeight / 2.0f;
	EKD = EyeKneeDistance;
	EKDD2 = EyeKneeDistance / 2.0f;

	if(Vertices != NULL && VerticesCount > 0)
	{
		TrianglesCount = VerticesCount / 3;

		Triangles = new CTriangle[TrianglesCount];

		for(int i = 0; i < TrianglesCount; i++)
		{
			Triangles[i].Set(Vertices[i * 3 + 0], Vertices[i * 3 + 1], Vertices[i * 3 + 2]);
		}
	}
}

void CCollisionDetector::Destroy()
{
	if(Triangles != NULL)
	{
		delete [] Triangles;
	}

	SetDefaults();
}

void CCollisionDetector::Jump()
{
	if(CrouchState == 0)
	{
		if(FallSpeed == 0.0f)
		{
			FallSpeed = -9.82f / 3.0f;
		}
	}
	else
	{
		CrouchState = 2;
	}
}

void CCollisionDetector::Crouch()
{
	if(CrouchState == 0)
	{
		EyeHeight = EHD2;
		EyeKneeDistance = EKDD2;
		CrouchState = 1;
	}
	else if(FallSpeed < 0.0f)
	{
		if(CrouchState == 1)
		{
			EyeHeight = EH;
			EyeKneeDistance = EKD;
			CrouchState = 0;
		}
	}
	else
	{
		if(CrouchState == 1)
		{
			CrouchState = 2;
		}
		else if(CrouchState == 2)
		{
			EyeHeight = EHD2;
			EyeKneeDistance = EKDD2;
			CrouchState = 1;
		}
	}
}

bool CCollisionDetector::GetHeightAbove(const vec3 &EyePositionA, float &MinDistance, float &Height)
{
	bool HeightFound = false;

	for(int i = 0; i < TrianglesCount; i++)
	{
		HeightFound |= Triangles[i].GetHeightAbove(EyePositionA, MinDistance, Height);
	}

	return HeightFound;
}

bool CCollisionDetector::GetHeightUnder(const vec3 &EyePositionA, float EyeKneeDistance, float &MinDistance, float &Height)
{
	bool HeightFound = false;

	for(int i = 0; i < TrianglesCount; i++)
	{
		HeightFound |= Triangles[i].GetHeightUnder(EyePositionA, EyeKneeDistance, MinDistance, Height);
	}

	return HeightFound;
}

bool CCollisionDetector::IntersectionTest(const vec3 &EyePositionA, const vec3 &EyePositionB, const vec3 &Direction, float EyeKneeDistance, float ClosestDistance, const vec3 &PN, float PD, float &MinDistance, vec3 &Compensation)
{
	bool IntersectionTestPassed = false;

	for(int i = 0; i < TrianglesCount; i++)
	{
		IntersectionTestPassed |= Triangles[i].IntersectionTest(EyePositionA, EyePositionB, Direction, EyeKneeDistance, ClosestDistance, PN, PD, MinDistance, Compensation);
	}

	return IntersectionTestPassed;
}

bool CCollisionDetector::DistanceTest(const vec3 &EyePositionB, float EyeKneeDistance, float ClosestDistance, float &MinDistance, vec3 &Compensation)
{
	bool DistanceTestFailed = false;

	for(int i = 0; i < TrianglesCount; i++)
	{
		DistanceTestFailed |= Triangles[i].DistanceTest(EyePositionB, EyeKneeDistance, ClosestDistance, MinDistance, Compensation);
	}

	return DistanceTestFailed;
}

void CCollisionDetector::CheckHorizontalCollision(const vec3 &EyePosition, vec3 &Movement)
{
	if(CrouchState != 0)
	{
		Movement *= 0.5f;
	}

	int Depth = 0;

	TestAgain:

	if(Depth < 16)
	{
		vec3 EyePositionA = EyePosition;
		float Length = length(Movement);
		vec3 Direction = Movement / Length;
		vec3 EyePositionB = EyePositionA + Movement;

		if(Length > ClosestDistance)
		{
			vec3 PN = cross(Direction, vec3(0.0f, -1.0f, 0.0f));
			float PD = -dot(PN, EyePositionA);
			float Distance = Length;
			vec3 Compensation;

			if(IntersectionTest(EyePositionA, EyePositionB, Direction, EyeKneeDistance, ClosestDistance, PN, PD, Distance, Compensation))
			{
				Movement += Compensation;

				Depth++;

				goto TestAgain;
			}
		}

		float Distance = ClosestDistance;
		vec3 Compensation;

		if(DistanceTest(EyePositionB, EyeKneeDistance, ClosestDistance, Distance, Compensation))
		{
			Movement += Compensation;

			Depth++;

			goto TestAgain;
		}
	}
}

void CCollisionDetector::CheckVerticalCollision(const vec3 &EyePosition, float FrameTime, vec3 &Movement)
{
	if(CrouchState == 2)
	{
		float DistanceAbove = EH - EyeHeight + ClosestDistance, HeightAbove;

		if(!GetHeightAbove(EyePosition, DistanceAbove, HeightAbove))
		{
			EyeHeight += EH * 2.0f * FrameTime;
			EyeKneeDistance = EyeHeight * EKD / EH;

			if(EyeHeight >= EH)
			{
				EyeHeight = EH;
				EyeKneeDistance = EKD;
				CrouchState = 0;
			}
		}
	}

	float DistanceUnder = 1048576.0f, HeightUnder = 0.0f;

	GetHeightUnder(EyePosition, EyeKneeDistance, DistanceUnder, HeightUnder);

	float EPYMEH = EyePosition.y - EyeHeight;

	if(HeightUnder < EPYMEH || FallSpeed < 0.0f)
	{
		FallSpeed += 9.82f * FrameTime;

		float Distance = FallSpeed * FrameTime;

		if(FallSpeed < 0.0f)
		{
			float DistanceAbove = ClosestDistance - Distance, HeightAbove;

			if(GetHeightAbove(EyePosition, DistanceAbove, HeightAbove))
			{
				Distance = DistanceAbove - ClosestDistance;
				FallSpeed = 0.0f;
			}
		}

		float EPYMEHMHU = EPYMEH - HeightUnder;

		if(Distance > EPYMEHMHU)
		{
			Distance = EPYMEHMHU;
		}

		Movement = vec3(0.0f, -Distance, 0.0f);
	}
	else
	{
		FallSpeed = 0.0f;

		float HUMEPYMEH = HeightUnder - EPYMEH;

		if(HUMEPYMEH < EyeHeight - EyeKneeDistance)
		{
			Movement = vec3(0.0f, HUMEPYMEH, 0.0f);
		}
	}

	if(Movement.y != 0.0f)
	{
		int Depth = 0;

		TestAgain:

		if(Depth < 16)
		{
			float Distance = ClosestDistance;
			vec3 Compensation;

			if(DistanceTest(EyePosition + Movement, EyeKneeDistance, ClosestDistance, Distance, Compensation))
			{
				Movement += Compensation;

				Depth++;

				goto TestAgain;
			}
		}
	}
}

// ----------------------------------------------------------------------------------------------------------------------------

CObject::CObject()
{
	sx = sy = sz = 1.0f;
	rx = ry = rz = tx = ty = tz = 0.0f;

	SetDefaults();
}

CObject::~CObject()
{
}

bool CObject::Load(const char *FileName)
{
	CString DirectoryFileName = ModuleDirectory + "Objects\\" + FileName;

	FILE *File;

	if(fopen_s(&File, DirectoryFileName, "rb") != 0)
	{
		ErrorLog.Append("Error loading file " + DirectoryFileName + "!\r\n");
		return false;
	}

	Destroy();

	this->FileName = new char[strlen(FileName) + 1];

	strcpy_s(this->FileName, strlen(FileName) + 1, FileName);

	char *Source;
	long SourceLength;

	fseek(File, 0, SEEK_END);
	SourceLength = ftell(File);
	fseek(File, 0, SEEK_SET);
	Source = new char[SourceLength + 1];
	fread(Source, 1, SourceLength, File);
	Source[SourceLength] = 0;

	fclose(File);

	for(long i = 0; i < SourceLength; i++)
	{
		if(Source[i] == '\r' || Source[i] == '\n') Source[i] = 0;
	}

	float si, ss, x, y, z, r, g, b, s, t;
	int v1, v2, v3, c1, c2, c3, tc1, tc2, tc3, n1, n2, n3;

	char *Line = Source;
	char *End = Source + SourceLength;

	while(Line < End)
	{
		while(Line < End && (*Line == ' ' || *Line == '\t')) Line++;

		if(Line[0] == '#')
		{
			// skip comment line
		}
		else if(sscanf_s(Line, "v %f %f %f", &x, &y, &z) == 3)
		{
			VerticesCount++;
		}
		else if(sscanf_s(Line, "c %f %f %f", &r, &g, &b) == 3)
		{
			ColorsCount++;
		}
		else if(sscanf_s(Line, "tc %f %f", &s, &t) == 2)
		{
			TexCoordsCount++;
		}
		else if(sscanf_s(Line, "n %f %f %f", &x, &y, &z) == 3)
		{
			NormalsCount++;
		}
		else if(sscanf_s(Line, "t %d/%d/%d/%d %d/%d/%d/%d %d/%d/%d/%d", &v1, &c1, &tc1, &n1, &v2, &c2, &tc2, &n2, &v3, &c3, &tc3, &n3) == 12)
		{
			if(v1 < 1 || v1 > VerticesCount || v2 < 1 || v2 > VerticesCount || v3 < 1 || v3 > VerticesCount) { ErrorLog.Append("Invalid vertex index!"); Destroy(); return false; }
			if(c1 < 0 || c1 > ColorsCount || c2 < 0 || c2 > ColorsCount || c3 < 0 || c3 > ColorsCount) { ErrorLog.Append("Invalid color index!"); Destroy(); return false; }
			if(tc1 < 0 || tc1 > TexCoordsCount || tc2 < 0 || tc2 > TexCoordsCount || tc3 < 0 || tc3 > TexCoordsCount) { ErrorLog.Append("Invalid texture coordinate index!"); Destroy(); return false; }
			if(n1 < 0 || n1 > NormalsCount || n2 < 0 || n2 > NormalsCount || n3 < 0 || n3 > NormalsCount) { ErrorLog.Append("Invalid normal index!"); Destroy(); return false; }

			TrianglesCount++;
		}

		while(Line < End && *Line != 0) Line++;
		while(Line < End && *Line == 0) Line++;
	}

	if(VerticesCount > 0) Vertices = new vec3[VerticesCount];
	if(ColorsCount > 0) Colors = new vec3[ColorsCount];
	if(TexCoordsCount > 0) TexCoords = new vec2[TexCoordsCount];
	if(NormalsCount > 0) Normals = new vec3[NormalsCount];

	if(TrianglesCount > 0) IndexArray = new CIndexArrayElement[TrianglesCount * 3];

	VerticesCount = ColorsCount = TexCoordsCount = NormalsCount = TrianglesCount = 0;

	Line = Source;

	while(Line < End)
	{
		while(Line < End && (*Line == ' ' || *Line == '\t')) Line++;

		if(Line[0] == '#')
		{
			// skip comment line
		}
		else if(Line[0] == 'd' && Line[1] == 'm' && (Line[2] == ' ' || Line[2] == '\t'))
        {
			char *DiffuseMapFileName = Line + 2;

			while(DiffuseMapFileName < End && (*DiffuseMapFileName == ' ' || *DiffuseMapFileName == '\t')) DiffuseMapFileName++;

			char *Character = DiffuseMapFileName;

			while(Character < End && *Character != 0 && *Character != ' ' && *Character != '\t') Character++;
			while(Character < End && *Character != 0) { *Character = 0; Character++; }

			if(!DiffuseMap.LoadTexture2D(DiffuseMapFileName))
			{
				return false;
			}
		}
		else if(Line[0] == 'n' && Line[1] == 'm' && (Line[2] == ' ' || Line[2] == '\t'))
        {
			char *NormalMapFileName = Line + 2;

			while(NormalMapFileName < End && (*NormalMapFileName == ' ' || *NormalMapFileName == '\t')) NormalMapFileName++;

			char *Character = NormalMapFileName;

			while(Character < End && *Character != 0 && *Character != ' ' && *Character != '\t') Character++;
			while(Character < End && *Character != 0) { *Character = 0; Character++; }

			if(!NormalMap.LoadTexture2D(NormalMapFileName))
			{
				return false;
			}
		}
		else if(sscanf_s(Line, "si %f", &si) == 1)
		{
			SpecularIntensity = si;
		}
		else if(sscanf_s(Line, "ss %f", &ss) == 1)
		{
			SpecularShininess = ss;
		}
		else if(sscanf_s(Line, "v %f %f %f", &x, &y, &z) == 3)
		{
			Vertices[VerticesCount++] = vec3(x, y, z);
		}
		else if(sscanf_s(Line, "c %f %f %f", &r, &g, &b) == 3)
		{
			Colors[ColorsCount++] = vec3(r, g, b);
		}
		else if(sscanf_s(Line, "tc %f %f", &s, &t) == 2)
		{
			TexCoords[TexCoordsCount++] = vec2(s, t);
		}
		else if(sscanf_s(Line, "n %f %f %f", &x, &y, &z) == 3)
		{
			Normals[NormalsCount++] = normalize(vec3(x, y, z));
		}
		else if(sscanf_s(Line, "t %d/%d/%d/%d %d/%d/%d/%d %d/%d/%d/%d", &v1, &c1, &tc1, &n1, &v2, &c2, &tc2, &n2, &v3, &c3, &tc3, &n3) == 12)
		{
			int ia = TrianglesCount * 3, ib = ia + 1, ic = ib + 1;

			IndexArray[ia].v = v1 - 1; IndexArray[ib].v = v2 - 1; IndexArray[ic].v = v3 - 1;
			IndexArray[ia].c = c1 - 1; IndexArray[ib].c = c2 - 1; IndexArray[ic].c = c3 - 1;
			IndexArray[ia].tc = tc1 - 1; IndexArray[ib].tc = tc2 - 1; IndexArray[ic].tc = tc3 - 1;
			IndexArray[ia].n = n1 - 1; IndexArray[ib].n = n2 - 1; IndexArray[ic].n = n3 - 1;

			TrianglesCount++;
		}

		while(Line < End && *Line != 0) Line++;
		while(Line < End && *Line == 0) Line++;
	}

	delete [] Source;

	/*GenerateVertexArrayAndVertexBufferObject();*/

	Triangles = new CTriangle[TrianglesCount];

	PrepareTriangles();

	return true;
}

void CObject::RenderIndexArray()
{
	glBegin(GL_TRIANGLES);

	for(int i = 0; i < TrianglesCount; i++)
	{
		int ia = i * 3, ib = ia + 1, ic = ib + 1;

		if(IndexArray[ia].c >= 0) glColor3fv(&Colors[IndexArray[ia].c]); else glColor3f(1.0f, 1.0f, 1.0f);
		if(IndexArray[ia].tc >= 0) glTexCoord2fv(&TexCoords[IndexArray[ia].tc]); else glTexCoord2f(0.0f, 0.0f);
		if(IndexArray[ia].n >= 0) glNormal3fv(&Normals[IndexArray[ia].n]); else glNormal3f(0.0f, 0.0f, 0.0f);
		glVertex3fv(&Vertices[IndexArray[ia].v]);

		if(IndexArray[ib].c >= 0) glColor3fv(&Colors[IndexArray[ib].c]); else glColor3f(1.0f, 1.0f, 1.0f);
		if(IndexArray[ib].tc >= 0) glTexCoord2fv(&TexCoords[IndexArray[ib].tc]); else glTexCoord2f(0.0f, 0.0f);
		if(IndexArray[ib].n >= 0) glNormal3fv(&Normals[IndexArray[ib].n]); else glNormal3f(0.0f, 0.0f, 0.0f);
		glVertex3fv(&Vertices[IndexArray[ib].v]);

		if(IndexArray[ic].c >= 0) glColor3fv(&Colors[IndexArray[ic].c]); else glColor3f(1.0f, 1.0f, 1.0f);
		if(IndexArray[ic].tc >= 0) glTexCoord2fv(&TexCoords[IndexArray[ic].tc]); else glTexCoord2f(0.0f, 0.0f);
		if(IndexArray[ic].n >= 0) glNormal3fv(&Normals[IndexArray[ic].n]); else glNormal3f(0.0f, 0.0f, 0.0f);
		glVertex3fv(&Vertices[IndexArray[ic].v]);
	}

	glEnd();
}

/*void CObject::RenderVertexArray(int TangentAttribLocation, int BitangentAttribLocation)
{
	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(3, GL_FLOAT, 68, (BYTE*)VertexArray + 0);

	glEnableClientState(GL_COLOR_ARRAY);
	glColorPointer(3, GL_FLOAT, 68, (BYTE*)VertexArray + 12);

	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	glTexCoordPointer(2, GL_FLOAT, 68, (BYTE*)VertexArray + 24);

	glEnableClientState(GL_NORMAL_ARRAY);
	glNormalPointer(GL_FLOAT, 68, (BYTE*)VertexArray + 32);

	if(TangentAttribLocation >= 0 && BitangentAttribLocation >= 0)
	{
		glEnableVertexAttribArray(TangentAttribLocation);
		glVertexAttribPointer(TangentAttribLocation, 3, GL_FLOAT, GL_FALSE, 68, (BYTE*)VertexArray + 44);

		glEnableVertexAttribArray(BitangentAttribLocation);
		glVertexAttribPointer(BitangentAttribLocation, 3, GL_FLOAT, GL_FALSE, 68, (BYTE*)VertexArray + 56);
	}

	glDrawArrays(GL_TRIANGLES, 0, TrianglesCount * 3);

	if(TangentAttribLocation >= 0 && BitangentAttribLocation >= 0)
	{
		glDisableVertexAttribArray(BitangentAttribLocation);
		glDisableVertexAttribArray(TangentAttribLocation);
	}

	glEnableClientState(GL_NORMAL_ARRAY);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	glEnableClientState(GL_COLOR_ARRAY);
	glEnableClientState(GL_VERTEX_ARRAY);
}*/

/*void CObject::RenderVertexBufferObject(int TangentAttribLocation, int BitangentAttribLocation)
{
	glBindBuffer(GL_ARRAY_BUFFER, VertexBufferObject);

	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(3, GL_FLOAT, 68, (void*)0);

	glEnableClientState(GL_COLOR_ARRAY);
	glColorPointer(3, GL_FLOAT, 68, (void*)12);

	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	glTexCoordPointer(2, GL_FLOAT, 68, (void*)24);

	glEnableClientState(GL_NORMAL_ARRAY);
	glNormalPointer(GL_FLOAT, 68, (void*)32);

	if(TangentAttribLocation >= 0 && BitangentAttribLocation >= 0)
	{
		glEnableVertexAttribArray(TangentAttribLocation);
		glVertexAttribPointer(TangentAttribLocation, 3, GL_FLOAT, GL_FALSE, 68, (void*)44);

		glEnableVertexAttribArray(BitangentAttribLocation);
		glVertexAttribPointer(BitangentAttribLocation, 3, GL_FLOAT, GL_FALSE, 68, (void*)56);
	}

	glDrawArrays(GL_TRIANGLES, 0, TrianglesCount * 3);

	if(TangentAttribLocation >= 0 && BitangentAttribLocation >= 0)
	{
		glDisableVertexAttribArray(BitangentAttribLocation);
		glDisableVertexAttribArray(TangentAttribLocation);
	}

	glEnableClientState(GL_NORMAL_ARRAY);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	glEnableClientState(GL_COLOR_ARRAY);
	glEnableClientState(GL_VERTEX_ARRAY);

	glBindBuffer(GL_ARRAY_BUFFER, 0);
}*/

void CObject::RenderVertices()
{
	glBegin(GL_POINTS);

	for(int i = 0; i < VerticesCount; i++)
	{
		glVertex3fv(&Vertices[i]);
	}

	glEnd();
}

void CObject::RenderEdges(int TriangleIndex)
{
	glBegin(GL_LINES);

	if(TriangleIndex == -1)
	{
		for(int i = 0; i < TrianglesCount; i++)
		{
			int ia = i * 3, ib = ia + 1, ic = ib + 1;

			glVertex3fv(&Vertices[IndexArray[ia].v]); glVertex3fv(&Vertices[IndexArray[ib].v]);
			glVertex3fv(&Vertices[IndexArray[ib].v]); glVertex3fv(&Vertices[IndexArray[ic].v]);
			glVertex3fv(&Vertices[IndexArray[ic].v]); glVertex3fv(&Vertices[IndexArray[ia].v]);
		}	
	}
	else
	{
		int ia = TriangleIndex * 3, ib = ia + 1, ic = ib + 1;

		glVertex3fv(&Vertices[IndexArray[ia].v]); glVertex3fv(&Vertices[IndexArray[ib].v]);
		glVertex3fv(&Vertices[IndexArray[ib].v]); glVertex3fv(&Vertices[IndexArray[ic].v]);
		glVertex3fv(&Vertices[IndexArray[ic].v]); glVertex3fv(&Vertices[IndexArray[ia].v]);
	}

	glEnd();
}

void CObject::RenderNormals()
{
	glBegin(GL_LINES);

	for(int i = 0; i < TrianglesCount; i++)
	{
		int ia = i * 3, ib = ia + 1, ic = ib + 1;

		bool HasPerVertexNormals = false;

		if(Normals[IndexArray[ia].n].x != Normals[IndexArray[ib].n].x || Normals[IndexArray[ib].n].x != Normals[IndexArray[ic].n].x)
		{
			HasPerVertexNormals = true;
		}
		else if(Normals[IndexArray[ia].n].y != Normals[IndexArray[ib].n].y || Normals[IndexArray[ib].n].y != Normals[IndexArray[ic].n].y)
		{
			HasPerVertexNormals = true;
		}
		else if(Normals[IndexArray[ia].n].z != Normals[IndexArray[ib].n].z || Normals[IndexArray[ib].n].z != Normals[IndexArray[ic].n].z)
		{
			HasPerVertexNormals = true;
		}

		vec3 VertexA = *(vec3*)&(ModelMatrix * vec4(Vertices[IndexArray[ia].v], 1.0f));
		vec3 VertexB = *(vec3*)&(ModelMatrix * vec4(Vertices[IndexArray[ib].v], 1.0f));
		vec3 VertexC = *(vec3*)&(ModelMatrix * vec4(Vertices[IndexArray[ic].v], 1.0f));

		if(HasPerVertexNormals)
		{
			glVertex3fv(&VertexA); glVertex3fv(&(VertexA + normalize(NormalMatrix * Normals[IndexArray[ia].n]) * 0.125f));
			glVertex3fv(&VertexB); glVertex3fv(&(VertexB + normalize(NormalMatrix * Normals[IndexArray[ib].n]) * 0.125f));
			glVertex3fv(&VertexC); glVertex3fv(&(VertexC + normalize(NormalMatrix * Normals[IndexArray[ic].n]) * 0.125f));
		}
		else
		{
			vec3 Middle = (VertexA + VertexB + VertexC) / 3.0f;

			glVertex3fv(&Middle); glVertex3fv(&(Middle + normalize(NormalMatrix * Normals[IndexArray[ia].n]) * 0.125f));
		}
	}

	glEnd();
}

void CObject::Destroy()
{
	delete [] FileName;

	DiffuseMap.Destroy();
	NormalMap.Destroy();

	delete [] Vertices;
	delete [] Colors;
	delete [] TexCoords;
	delete [] Normals;

	delete [] IndexArray;
	delete [] VertexArray;

	glDeleteBuffers(1, &VertexBufferObject);

	delete [] Triangles;

	SetDefaults();
}

void CObject::CalculateModelMatrix()
{
	SetModelMatrix(translate(tx, ty, tz) * rotate(rz, vec3(0.0f, 0.0f, 1.0f)) * rotate(ry, vec3(0.0f, 1.0f, 0.0f)) * rotate(rx, vec3(1.0f, 0.0f, 0.0f)) * scale(sx, sy, sz));
}

char* CObject::GetFileName()
{
	return FileName;
}

void CObject::SetModelMatrix(const mat4x4 &ModelMatrix)
{
	this->ModelMatrix = ModelMatrix;

	TangentMatrix = mat3x3(ModelMatrix);
	NormalMatrix = transpose(inverse(TangentMatrix));

	PrepareTriangles();
}

mat4x4& CObject::GetModelMatrix()
{
	return ModelMatrix;
}

mat3x3& CObject::GetTangentMatrix()
{
	return TangentMatrix;
}

mat3x3& CObject::GetNormalMatrix()
{
	return NormalMatrix;
}

GLuint CObject::GetDiffuseMap()
{
	return DiffuseMap;
}

GLuint CObject::GetNormalMap()
{
	return NormalMap;
}

float CObject::GetSpecularIntensity()
{
	return SpecularIntensity;
}

float CObject::GetSpecularShininess()
{
	return SpecularShininess;
}

CTriangle* CObject::GetTriangles()
{
	return Triangles;
}

int CObject::GetTrianglesCount()
{
	return TrianglesCount;
}

vec3 CObject::GetMiddle()
{
	vec3 Middle;

	if(VerticesCount > 0)
	{
		vec3 Vertex, Min, Max;

		Vertex = *(vec3*)&(ModelMatrix * vec4(Vertices[0], 1.0f));

		Min = Max = Vertex;

		for(int i = 1; i < VerticesCount; i++)
		{
			Vertex = *(vec3*)&(ModelMatrix * vec4(Vertices[i], 1.0f));

			if(Vertex.x > Min.x) Min.x = Vertex.x;
			if(Vertex.y > Min.y) Min.y = Vertex.y;
			if(Vertex.z > Min.z) Min.z = Vertex.z;

			if(Vertex.x < Max.x) Max.x = Vertex.x;
			if(Vertex.y < Max.y) Max.y = Vertex.y;
			if(Vertex.z < Max.z) Max.z = Vertex.z;
		}

		Middle = (Min + Max) / 2.0f;
	}

	return Middle;
}

bool CObject::RayTrianglesIntersectionTest(const vec3 &RayOrigin, const vec3 &RayDirection, float &MinDistance, vec3 &IntersectionPoint, int &TriangleIndex)
{
	bool TriangleFound = false;

	for(int i = 0; i < TrianglesCount; i++)
	{
		if(Triangles[i].RayTriangleIntersectionTest(RayOrigin, RayDirection, MinDistance, IntersectionPoint))
		{
			TriangleFound = true;
			TriangleIndex = i;
		}
	}

	return TriangleFound;
}

void CObject::PrepareTriangles()
{
	for(int i = 0; i < TrianglesCount; i++)
	{
		int ia = i * 3, ib = ia + 1, ic = ib + 1;

		vec3 A = *(vec3*)&(ModelMatrix * vec4(Vertices[IndexArray[ia].v], 1.0f));
		vec3 B = *(vec3*)&(ModelMatrix * vec4(Vertices[IndexArray[ib].v], 1.0f));
		vec3 C = *(vec3*)&(ModelMatrix * vec4(Vertices[IndexArray[ic].v], 1.0f));

		Triangles[i].Set(A, B, C);
	}
}

/*void CObject::GenerateVertexArrayAndVertexBufferObject()
{
	VertexArray = new CVertexArrayElement[TrianglesCount * 3];

	for(int i = 0; i < TrianglesCount; i++)
	{
		int ia = i * 3, ib = ia + 1, ic = ib + 1;

		VertexArray[ia].Vertex = Vertices[IndexArray[ia].v];
		if(IndexArray[ia].c >= 0) VertexArray[ia].Color = Colors[IndexArray[ia].c]; else VertexArray[ia].Color = vec3(1.0f, 1.0f, 1.0f);
		if(IndexArray[ia].tc >= 0) VertexArray[ia].TexCoord = TexCoords[IndexArray[ia].tc];
		if(IndexArray[ia].n >=0) VertexArray[ia].Normal = Normals[IndexArray[ia].n];

		VertexArray[ib].Vertex = Vertices[IndexArray[ib].v];
		if(IndexArray[ib].c >= 0) VertexArray[ib].Color = Colors[IndexArray[ib].c]; else VertexArray[ib].Color = vec3(1.0f, 1.0f, 1.0f);
		if(IndexArray[ib].tc >= 0) VertexArray[ib].TexCoord = TexCoords[IndexArray[ib].tc];
		if(IndexArray[ib].n >= 0) VertexArray[ib].Normal = Normals[IndexArray[ib].n];

		VertexArray[ic].Vertex = Vertices[IndexArray[ic].v];
		if(IndexArray[ic].c >= 0) VertexArray[ic].Color = Colors[IndexArray[ic].c]; else VertexArray[ic].Color = vec3(1.0f, 1.0f, 1.0f);
		if(IndexArray[ic].tc >= 0) VertexArray[ic].TexCoord = TexCoords[IndexArray[ic].tc];
		if(IndexArray[ic].n >= 0) VertexArray[ic].Normal = Normals[IndexArray[ic].n];

		vec3 vdab = VertexArray[ib].Vertex - VertexArray[ia].Vertex;
		vec3 vdac = VertexArray[ic].Vertex - VertexArray[ia].Vertex;

		vec2 tcdab = VertexArray[ib].TexCoord - VertexArray[ia].TexCoord;
		vec2 tcdac = VertexArray[ic].TexCoord - VertexArray[ia].TexCoord;

		float r = 1.0f / (tcdab.x * tcdac.y - tcdab.y * tcdac.x);

		vec3 Normal = normalize(cross(vdab, vdac));
		vec3 Tangent = normalize((vdab * tcdac.y  - vdac * tcdab.y) * r);
		vec3 Bitangent = normalize((vdac * tcdab.x  - vdab * tcdac.x) * r);

		if(VertexArray[ia].Normal.x == 0.0 && VertexArray[ia].Normal.y == 0.0 && VertexArray[ia].Normal.z == 0.0)
		{
			VertexArray[ia].Normal = Normal;
			VertexArray[ia].Tangent = Tangent;
			VertexArray[ia].Bitangent = Bitangent;
		}
		else
		{
			VertexArray[ia].Normal = normalize(VertexArray[ia].Normal);
			VertexArray[ia].Tangent = normalize(Tangent - VertexArray[ia].Normal * dot(VertexArray[ia].Normal, Tangent));
			VertexArray[ia].Bitangent = cross(VertexArray[ia].Normal, VertexArray[ia].Tangent);
	    }

		if(VertexArray[ib].Normal.x == 0.0 && VertexArray[ib].Normal.y == 0.0 && VertexArray[ib].Normal.z == 0.0)
		{
			VertexArray[ib].Normal = Normal;
			VertexArray[ib].Tangent = Tangent;
			VertexArray[ib].Bitangent = Bitangent;
		}
		else
		{
			VertexArray[ib].Normal = normalize(VertexArray[ib].Normal);
			VertexArray[ib].Tangent = normalize(Tangent - VertexArray[ib].Normal * dot(VertexArray[ib].Normal, Tangent));
			VertexArray[ib].Bitangent = cross(VertexArray[ib].Normal, VertexArray[ib].Tangent);
        }

		if(VertexArray[ic].Normal.x == 0.0 && VertexArray[ic].Normal.y == 0.0 && VertexArray[ic].Normal.z == 0.0)
		{
			VertexArray[ic].Normal = Normal;
			VertexArray[ic].Tangent = Tangent;
			VertexArray[ic].Bitangent = Bitangent;
		}
		else
		{
			VertexArray[ic].Normal = normalize(VertexArray[ic].Normal);
			VertexArray[ic].Tangent = normalize(Tangent - VertexArray[ic].Normal * dot(VertexArray[ic].Normal, Tangent));
			VertexArray[ic].Bitangent = cross(VertexArray[ic].Normal, VertexArray[ia].Tangent);
	    }
	}

	glGenBuffers(1, &VertexBufferObject);

	glBindBuffer(GL_ARRAY_BUFFER, VertexBufferObject);
	glBufferData(GL_ARRAY_BUFFER, TrianglesCount * 3 * 68, VertexArray, GL_STATIC_DRAW);
	glBindBuffer(GL_ARRAY_BUFFER, 0);
}*/

void CObject::SetDefaults()
{
	FileName = NULL;

	ModelMatrix = mat4x4();
	TangentMatrix = mat3x3();
	NormalMatrix = mat3x3();

	VerticesCount = ColorsCount = TexCoordsCount = NormalsCount = TrianglesCount = 0;

	SpecularIntensity = SpecularShininess = 0.0f;

	Vertices = NULL;
	Colors = NULL;
	TexCoords = NULL;
	Normals = NULL;

	IndexArray = NULL;

	VertexArray = NULL;
	VertexBufferObject = 0;

	Triangles = NULL;
}

// ----------------------------------------------------------------------------------------------------------------------------

CScene::CScene()
{
	SetDefaults();
}

CScene::~CScene()
{
}

bool CScene::Load(const char *FileName)
{
	CString DirectoryFileName = ModuleDirectory + "Scenes\\" + FileName;

	FILE *File;

	if(fopen_s(&File, DirectoryFileName, "rb") != 0)
	{
		ErrorLog.Append("Error loading file " + DirectoryFileName + "!\r\n");
		return false;
	}

	char *Source;
	long SourceLength;

	fseek(File, 0, SEEK_END);
	SourceLength = ftell(File);
	fseek(File, 0, SEEK_SET);
	Source = new char[SourceLength + 1];
	fread(Source, 1, SourceLength, File);
	Source[SourceLength] = 0;

	fclose(File);

	for(long i = 0; i < SourceLength; i++)
	{
		if(Source[i] == '\r' || Source[i] == '\n') Source[i] = 0;
	}

	float sx, sy, sz, rx, ry, rz, tx, ty, tz;

	char *Line = Source;
	char *End = Source + SourceLength;

	while(Line < End)
	{
		while(Line < End && (*Line == ' ' || *Line == '\t')) Line++;

		if(Line[0] == '#')
		{
			// skip comment line
		}
		else if(Line[0] == 'o')
		{
			char *ObjectFileName = Line + 1;

			while(ObjectFileName < End && (*ObjectFileName == ' ' || *ObjectFileName == '\t')) ObjectFileName++;

			char *Character = ObjectFileName;

			while(Character < End && *Character != 0 && *Character != ' ' && *Character != '\t') Character++;
			while(Character < End && *Character != 0 && (*Character == ' ' || *Character == '\t')) Character++;

			Line = Character;

			if(sscanf_s(Line, "%f %f %f %f %f %f %f %f %f", &sx, &sy, &sz, &rx, &ry, &rz, &tx, &ty, &tz) == 9)
			{
				ObjectsCount++;
			}
		}

		while(Line < End && *Line != 0) Line++;
		while(Line < End && *Line == 0) Line++;
	}

	if(ObjectsCount > 0) Objects = new CObject[ObjectsCount];

	ObjectsCount = 0;

	Line = Source;

	while(Line < End)
	{
		while(Line < End && (*Line == ' ' || *Line == '\t')) Line++;

		if(Line[0] == '#')
		{
			// skip comment line
		}
		else if(Line[0] == 'o')
		{
			char *ObjectFileName = Line + 1;

			while(ObjectFileName < End && (*ObjectFileName == ' ' || *ObjectFileName == '\t')) ObjectFileName++;

			char *Character = ObjectFileName;

			while(Character < End && *Character != 0 && *Character != ' ' && *Character != '\t') Character++;
			while(Character < End && *Character != 0 && (*Character == ' ' || *Character == '\t')) { *Character = 0; Character++; }

			Line = Character;

			if(sscanf_s(Line, "%f %f %f %f %f %f %f %f %f", &sx, &sy, &sz, &rx, &ry, &rz, &tx, &ty, &tz) == 9)
			{
				int Index = ObjectsCount;

				Objects[Index].Load(ObjectFileName);

				Objects[Index].sx = sx;
				Objects[Index].sy = sy;
				Objects[Index].sz = sz;
				Objects[Index].rx = rx;
				Objects[Index].ry = ry;
				Objects[Index].rz = rz;
				Objects[Index].tx = tx;
				Objects[Index].ty = ty;
				Objects[Index].tz = tz;

				Objects[Index].CalculateModelMatrix();

				ObjectsCount++;
			}
		}

		while(Line < End && *Line != 0) Line++;
		while(Line < End && *Line == 0) Line++;
	}

	delete [] Source;

	return true;
}

bool CScene::Save(const char *FileName)
{
	CString DirectoryFileName = ModuleDirectory + "Scenes\\" + FileName;

	FILE *File;

	if(fopen_s(&File, DirectoryFileName, "wb+") != 0)
	{
		ErrorLog.Append("Error creating file " + DirectoryFileName + "!\r\n");
		return false;
	}

	for(int i = 0; i < ObjectsCount; i++)
	{
		fprintf(File, "o %s %f %f %f %f %f %f %f %f %f\r\n", Objects[i].GetFileName(), Objects[i].sx, Objects[i].sy, Objects[i].sz, Objects[i].rx, Objects[i].ry, Objects[i].rz, Objects[i].tx, Objects[i].ty, Objects[i].tz);
	}

	fclose(File);

	return true;
}

bool CScene::SaveBinary(const char *FileName)
{
	CString DirectoryFileName = ModuleDirectory + "Scenes\\" + FileName;

	FILE *File;

	if(fopen_s(&File, DirectoryFileName, "wb+") != 0)
	{
		ErrorLog.Append("Error creating file " + DirectoryFileName + "!\r\n");
		return false;
	}

	int VerticesCount = 0;

	for(int i = 0; i < ObjectsCount; i++)
	{
		VerticesCount += Objects[i].GetTrianglesCount() * 3;
	}

	fwrite(&VerticesCount, sizeof(int), 1, File);

	for(int i = 0; i < ObjectsCount; i++)
	{
		CTriangle *Triangles = Objects[i].GetTriangles();
		int TrianglesCount = Objects[i].GetTrianglesCount();
		int VerticesCount = TrianglesCount * 3;

		vec3 *Vertices = new vec3[VerticesCount];

		for(int i = 0; i < TrianglesCount; i++)
		{
			int ia = i * 3, ib = ia + 1, ic = ib + 1;

			Vertices[ia] = Triangles[i].A;
			Vertices[ib] = Triangles[i].B;
			Vertices[ic] = Triangles[i].C;
		}

		fwrite(Vertices, sizeof(vec3), VerticesCount, File);

		delete [] Vertices;
	}

	fclose(File);

	return true;
}

void CScene::RenderIndexArray(int ObjectIndex)
{
	Objects[ObjectIndex].RenderIndexArray();
}

void CScene::RenderVertices(int ObjectIndex)
{
	Objects[ObjectIndex].RenderVertices();
}

void CScene::RenderEdges(int ObjectIndex)
{
	Objects[ObjectIndex].RenderEdges();
}

void CScene::RenderNormals(int ObjectIndex)
{
	Objects[ObjectIndex].RenderNormals();
}

void CScene::Destroy()
{
	for(int i = 0; i < ObjectsCount; i++)
	{
		Objects[i].Destroy();
	}

	delete [] Objects;

	SetDefaults();
}

int CScene::GetObjectsCount()
{
	return ObjectsCount;
}

CTriangle* CScene::GetTriangles(int ObjectID)
{
	return Objects[ObjectID].GetTriangles();
}

int CScene::GetTrianglesCount(int ObjectID)
{
	return Objects[ObjectID].GetTrianglesCount();
}

mat4x4& CScene::GetModelMatrix(int ObjectIndex)
{
	return Objects[ObjectIndex].GetModelMatrix();
}

mat3x3& CScene::GetTangentMatrix(int ObjectIndex)
{
	return Objects[ObjectIndex].GetTangentMatrix();
}

mat3x3& CScene::GetNormalMatrix(int ObjectIndex)
{
	return Objects[ObjectIndex].GetNormalMatrix();
}

vec3 CScene::GetMiddle()
{
	vec3 Middle;

	if(ObjectsCount > 0)
	{
		for(int i = 0; i < ObjectsCount; i++)
		{
			Middle += Objects[i].GetMiddle();
		}

		Middle /= (float)ObjectsCount;
	}

	return Middle;
}

void CScene::AddObject(CObject &Object)
{
	CObject *NewObjects = new CObject[ObjectsCount + 1];

	for(int i = 0; i < ObjectsCount; i++)
	{
		NewObjects[i] = Objects[i];
	}

	delete [] Objects;

	Objects = NewObjects;

	int Index = ObjectsCount;

	Objects[Index].Load(Object.GetFileName());

	Objects[Index].sx = Object.sx;
	Objects[Index].sy = Object.sy;
	Objects[Index].sz = Object.sz;
	Objects[Index].rx = Object.rx;
	Objects[Index].ry = Object.ry;
	Objects[Index].rz = Object.rz;
	Objects[Index].tx = Object.tx;
	Objects[Index].ty = Object.ty;
	Objects[Index].tz = Object.tz;

	Objects[Index].CalculateModelMatrix();

	ObjectsCount++;
}

void CScene::DeleteObject(int ObjectIndex)
{
	if(ObjectIndex >= 0 && ObjectIndex < ObjectsCount)
	{
		Objects[ObjectIndex].Destroy();

		for(int i = ObjectIndex; i < ObjectsCount - 1; i++)
		{
			Objects[i] = Objects[i + 1];
		}

		ObjectsCount--;

		if(ObjectsCount > 0)
		{
			CObject *NewObjects = new CObject[ObjectsCount];

			for(int i = 0; i < ObjectsCount; i++)
			{
				NewObjects[i] = Objects[i];
			}

			delete [] Objects;

			Objects = NewObjects;
		}
		else
		{
			delete [] Objects;
			Objects = NULL;
		}
	}
}

bool CScene::RayTrianglesIntersectionTest(const vec3 &RayOrigin, const vec3 &RayDirection, float &MinDistance, vec3 &IntersectionPoint, int &TriangleIndex, int &ObjectIndex)
{
	bool ObjectFound = false;

	for(int i = 0; i < ObjectsCount; i++)
	{
		if(Objects[i].RayTrianglesIntersectionTest(RayOrigin, RayDirection, MinDistance, IntersectionPoint, TriangleIndex))
		{
			ObjectFound = true;
			ObjectIndex = i;
		}
	}

	return ObjectFound;
}

void CScene::SetDefaults()
{
	Objects = NULL;
	ObjectsCount = 0;
}

// ----------------------------------------------------------------------------------------------------------------------------

bool SceneFileExists(int SceneId)
{
	CString FileName;

	FileName.Set("scene%d.scn", SceneId);

	CString DirectoryFileName = ModuleDirectory + "Scenes\\" + FileName;

	FILE *File;

	if(fopen_s(&File, DirectoryFileName, "rb") == 0)
	{
		fclose(File);

		return true;
	}

	return false;
}

bool ObjectFileExists(int ObjectId)
{
	CString FileName;

	FileName.Set("object%d.objs", ObjectId);

	CString DirectoryFileName = ModuleDirectory + "Objects\\" + FileName;

	FILE *File;

	if(fopen_s(&File, DirectoryFileName, "rb") == 0)
	{
		fclose(File);

		return true;
	}

	return false;
}

// ----------------------------------------------------------------------------------------------------------------------------

COpenGLRenderer::COpenGLRenderer()
{
	EditMode = 1;

	IntersectionPlaneNormal = vec3(0.0f, 1.0f, 0.0f);
	IntersectionPlaneD = 0.0f;
	IntersectionPointRoundFactor = 0.125f;

	RenderVertices = true;
	RenderEdges = true;
	RenderNormals = false;
	ShowAxisGrid = true;
}

COpenGLRenderer::~COpenGLRenderer()
{
}

bool COpenGLRenderer::Init()
{
	bool Error = false;

	Error |= !Lighting.Load("lighting.vs", "lighting.fs");

	Lighting.UniformLocations = new GLuint[3];
	Lighting.UniformLocations[0] = glGetUniformLocation(Lighting, "ModelMatrix");
	Lighting.UniformLocations[1] = glGetUniformLocation(Lighting, "NormalMatrix");
	Lighting.UniformLocations[2] = glGetUniformLocation(Lighting, "CameraPosition");

	if(Error)
	{
		return false;
	}

	SceneId = 1;

	SceneFileName.Set("scene%d.scn", SceneId);

	Scene.Load(SceneFileName);

	ObjectId = 1;

	ObjectFileName.Set("object%d.objs", ObjectId);

	Object.Load(ObjectFileName);

	Text = SceneFileName + ", " + ObjectFileName;

	Camera.Look(vec3(3.0f, 3.0f, 3.0f), vec3(0.0f, 0.0f, 0.0f), true);

	return true;
}

void COpenGLRenderer::Render(float FrameTime)
{
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glEnable(GL_DEPTH_TEST);
	glEnable(GL_CULL_FACE);

	glMatrixMode(GL_MODELVIEW);
	glLoadMatrixf(&Camera.ViewMatrix);

	float d = 50.0f;

	if(ShowAxisGrid && EditMode != 3)
	{
		glLineWidth(2.0f);

		glBegin(GL_LINES);

		glColor3f(1.0f, 0.0f, 0.0f);

		glVertex3f(0.0f, 0.0f, 0.0f); glVertex3f(1.0f, 0.0f, 0.0f);
		glVertex3f(1.0f, 0.1f, 0.0f); glVertex3f(1.1f, -0.1f, 0.0f);
		glVertex3f(1.1f, 0.1f, 0.0f); glVertex3f(1.0f, -0.1f, 0.0f);

		glColor3f(0.0f, 1.0f, 0.0f);

		glVertex3f(0.0f, 0.0f, 0.0f); glVertex3f(0.0f, 1.0f, 0.0f);
		glVertex3f(-0.05f, 1.25f, 0.0f); glVertex3f(0.0f, 1.15f, 0.0f);
		glVertex3f(0.05f,1.25f, 0.0f); glVertex3f(0.0f, 1.15f, 0.0f);
		glVertex3f(0.0f,1.15f, 0.0f); glVertex3f(0.0f, 1.05f, 0.0f);

		glColor3f(0.0f, 0.0f, 1.0f);

		glVertex3f(0.0f,0.0f,0.0f); glVertex3f(0.0f, 0.0f, 1.0f);
		glVertex3f(-0.05f,0.1f,1.05f); glVertex3f(0.05f, 0.1f, 1.05f);
		glVertex3f(0.05f,0.1f,1.05f); glVertex3f(-0.05f, -0.1f, 1.05f);
		glVertex3f(-0.05f,-0.1f,1.05f); glVertex3f(0.05f, -0.1f, 1.05f);

		glEnd();

		glLineWidth(1.0f);

		glColor3f(1.0f, 1.0f, 1.0f);

		glBegin(GL_LINES);

		if(IntersectionPlaneNormal.x == 1.0f)
		{
			for(float i = -d; i <= d; i += 1.0f)
			{
				glVertex3f(-IntersectionPlaneD,  i, -d);
				glVertex3f(-IntersectionPlaneD,  i,  d);
				glVertex3f(-IntersectionPlaneD, -d,  i);
				glVertex3f(-IntersectionPlaneD,  d,  i);
			}
		}

		if(IntersectionPlaneNormal.y == 1.0f)
		{
			for(float i = -d; i <= d; i += 1.0f)
			{
				glVertex3f( i, -IntersectionPlaneD, -d);
				glVertex3f( i, -IntersectionPlaneD,  d);
				glVertex3f(-d, -IntersectionPlaneD,  i);
				glVertex3f( d, -IntersectionPlaneD,  i);
			}
		}

		if(IntersectionPlaneNormal.z == 1.0f)
		{
			for(float i = -d; i <= d; i += 1.0f)
			{
				glVertex3f( i, -d, -IntersectionPlaneD);
				glVertex3f( i,  d, -IntersectionPlaneD);
				glVertex3f(-d,  i, -IntersectionPlaneD);
				glVertex3f( d,  i, -IntersectionPlaneD);
			}
		}

		glEnd();
	}

	for(int i = 0; i < Scene.GetObjectsCount(); i++)
	{
		glMatrixMode(GL_MODELVIEW);
		glLoadMatrixf(&Camera.ViewMatrix);

		glMultMatrixf(&Scene.GetModelMatrix(i));

		glUseProgram(Lighting);

		glUniformMatrix4fv(Lighting.UniformLocations[0], 1, GL_FALSE, &Scene.GetModelMatrix(i));
		glUniformMatrix3fv(Lighting.UniformLocations[1], 1, GL_FALSE, &Scene.GetNormalMatrix(i));
		glUniform3fv(Lighting.UniformLocations[2], 1, &Camera.Position);

		Scene.RenderIndexArray(i);

		glUseProgram(0);

		if(RenderVertices && EditMode != 3)
		{
			glPointSize(3.0f);
			glColor3f(0.0f, 0.5f, 1.0f);
			Scene.RenderVertices(i);
			glPointSize(1.0f);
		}

		if(RenderEdges && EditMode != 3)
		{
			glColor3f(0.0f, 0.0f, 0.0f);
			Scene.RenderEdges(i);
		}

		if(RenderNormals && EditMode != 3)
		{
			glMatrixMode(GL_MODELVIEW);
			glLoadMatrixf(&Camera.ViewMatrix);
			glColor3f(0.0f, 0.0f, 1.0f);
			Scene.RenderNormals(i);
		}
	}

	if(EditMode == 1 && IntersectionPointFound)
	{
		glMatrixMode(GL_MODELVIEW);
		glLoadMatrixf(&Camera.ViewMatrix);

		glMultMatrixf(&Object.GetModelMatrix());

		glUseProgram(Lighting);

		glUniformMatrix4fv(Lighting.UniformLocations[0], 1, GL_FALSE, &Object.GetModelMatrix());
		glUniformMatrix3fv(Lighting.UniformLocations[1], 1, GL_FALSE, &Object.GetNormalMatrix());
		glUniform3fv(Lighting.UniformLocations[2], 1, &Camera.Position);

		Object.RenderIndexArray();

		glUseProgram(0);

		if(RenderVertices)
		{
			glPointSize(3.0f);
			glColor3f(0.0f, 0.5f, 1.0f);
			Object.RenderVertices();
			glPointSize(1.0f);
		}

		if(RenderEdges)
		{
			glColor3f(0.0f, 0.0f, 0.0f);
			Object.RenderEdges();
		}

		if(RenderNormals)
		{
			glMatrixMode(GL_MODELVIEW);
			glLoadMatrixf(&Camera.ViewMatrix);
			glColor3f(0.0f, 0.0f, 1.0f);
			Object.RenderNormals();
		}
	}

	glBindTexture(GL_TEXTURE_2D, 0);

	if(EditMode == 2 && ObjectFound)
	{
		glMatrixMode(GL_MODELVIEW);
		glLoadMatrixf(&Camera.ViewMatrix);
		glMultMatrixf(&Scene.GetModelMatrix(Index));
		glLineWidth(3.0f);
		glColor3f(0.0f, 1.0f, 0.0f);
		Scene.RenderEdges(Index);
		glLineWidth(1.0f);
	}

	glDisable(GL_CULL_FACE);

	glMatrixMode(GL_MODELVIEW);
	glLoadMatrixf(&Camera.ViewMatrix);

	if(ShowAxisGrid && EditMode != 3)
	{
		glColor4f(0.0, 1.0f, 0.0f, 0.25f);

		glEnable(GL_BLEND);
		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

		glBegin(GL_QUADS);

		if(IntersectionPlaneNormal.x == 1.0f)
		{
			glVertex3f(-IntersectionPlaneD, -d,  d);
			glVertex3f(-IntersectionPlaneD, -d, -d);
			glVertex3f(-IntersectionPlaneD,  d, -d);
			glVertex3f(-IntersectionPlaneD,  d,  d);
		}

		if(IntersectionPlaneNormal.y == 1.0f)
		{
			glVertex3f(-d, -IntersectionPlaneD,  d);
			glVertex3f( d, -IntersectionPlaneD,  d);
			glVertex3f( d, -IntersectionPlaneD, -d);
			glVertex3f(-d, -IntersectionPlaneD, -d);
		}

		if(IntersectionPlaneNormal.z == 1.0f)
		{
			glVertex3f(-d, -d, -IntersectionPlaneD);
			glVertex3f( d, -d, -IntersectionPlaneD);
			glVertex3f( d,  d, -IntersectionPlaneD);
			glVertex3f(-d,  d, -IntersectionPlaneD);
		}

		glEnd();

		glDisable(GL_BLEND);
	}

	glDisable(GL_DEPTH_TEST);
}

void COpenGLRenderer::Resize(int Width, int Height)
{
	this->Width = Width;
	this->Height = Height;

	glViewport(0, 0, Width, Height);

	Camera.SetPerspective(45.0f, (float)Width / (float)Height, 0.125f, 512.0f);

	glMatrixMode(GL_PROJECTION);
	glLoadMatrixf(&Camera.ProjectionMatrix);
}

void COpenGLRenderer::Destroy()
{
	Lighting.Destroy();

	Scene.Destroy();

	Object.Destroy();

	CollisionDetector.Destroy();
}

void COpenGLRenderer::InitCollisionDetector()
{
	int ObjectsCount = Scene.GetObjectsCount(), TrianglesCount = 0;

	for(int i = 0; i < ObjectsCount; i++)
	{
		TrianglesCount += Scene.GetTrianglesCount(i);
	}

	int VerticesCount = TrianglesCount * 3;

	vec3 *Vertices = new vec3[VerticesCount];

	int VertexIndex = 0;

	for(int i = 0; i < ObjectsCount; i++)
	{
		CTriangle *Triangles = Scene.GetTriangles(i);
		TrianglesCount = Scene.GetTrianglesCount(i);

		for(int i = 0; i < TrianglesCount; i++)
		{
			Vertices[VertexIndex++] = Triangles[i].A;
			Vertices[VertexIndex++] = Triangles[i].B;
			Vertices[VertexIndex++] = Triangles[i].C;
		}
	}

	CollisionDetector.Init(Vertices, VerticesCount, 1.75f, 1.25f, 0.125f);

	delete [] Vertices;
}

void COpenGLRenderer::CheckCameraKeys(float FrameTime)
{
	BYTE Keys = 0x00;

	if(GetKeyState('W') & 0x80) Keys |= 0x01;
	if(GetKeyState('S') & 0x80) Keys |= 0x02;
	if(GetKeyState('A') & 0x80) Keys |= 0x04;
	if(GetKeyState('D') & 0x80) Keys |= 0x08;

	if(EditMode != 3)
	{
		if(GetKeyState('R') & 0x80) Keys |= 0x10;
		if(GetKeyState('F') & 0x80) Keys |= 0x20;
	}

	if(GetKeyState(VK_SHIFT) & 0x80) Keys |= 0x40;
	if(GetKeyState(VK_CONTROL) & 0x80) Keys |= 0x80;

	if(Keys & 0x3F)
	{
		vec3 Movement = Camera.OnKeys(Keys, EditMode == 3 ? FrameTime * 0.5f : FrameTime);

		if(EditMode == 3)
		{
			CollisionDetector.CheckHorizontalCollision(Camera.Reference, Movement);
		}

		if(length(Movement) > 0.0f)
		{
			Camera.Move(Movement);
		}
	}

	if(EditMode == 3)
	{
		vec3 Movement;

		CollisionDetector.CheckVerticalCollision(Camera.Reference, FrameTime, Movement);

		if(length(Movement) > 0.0f)
		{
			Camera.Move(Movement);
		}
	}

	FindPrimitive();
}

int round(float number)
{
	return number >= 0.0f ? (int)(number + 0.5f) : (int)(number - 0.5f);
}

void COpenGLRenderer::FindPrimitive()
{
	vec4 Point = vec4(((float)LastX / Width) * 2.0f - 1.0f, ((float)(Height - 1 - LastY) / Height) * 2.0f - 1.0f, 0.0f, 1.0f);

	Point = Camera.ViewProjectionMatrixInverse * Point;
	Point /= Point.w;

	vec3 Ray = normalize(*(vec3*)&Point - Camera.Position);

	if(EditMode == 1)
	{
		IntersectionPointFound = false;

		float NdotR = -dot(IntersectionPlaneNormal, Ray);

		if(NdotR != 0.0f)
		{
			float Distance = (dot(IntersectionPlaneNormal, Camera.Position) + IntersectionPlaneD) / NdotR;

			if(Distance > 0.125f)
			{
				IntersectionPoint = Camera.Position + Ray * Distance;

				IntersectionPoint.x = round(IntersectionPoint.x / IntersectionPointRoundFactor) * IntersectionPointRoundFactor;
				IntersectionPoint.y = round(IntersectionPoint.y / IntersectionPointRoundFactor) * IntersectionPointRoundFactor;
				IntersectionPoint.z = round(IntersectionPoint.z / IntersectionPointRoundFactor) * IntersectionPointRoundFactor;

				IntersectionPointFound = true;
			}
		}

		if(IntersectionPointFound)
		{
			Object.tx = IntersectionPoint.x;
			Object.ty = IntersectionPoint.y;
			Object.tz = IntersectionPoint.z;

			Object.CalculateModelMatrix();
		}
	}

	if(EditMode == 2)
	{
		ObjectFound = false;

		float Distance = 1048576.0f;
		vec3 IntersectionPoint;
		int TriangleIndex, ObjectIndex;

		if(Scene.RayTrianglesIntersectionTest(Camera.Position, Ray, Distance, IntersectionPoint, TriangleIndex, ObjectIndex))
		{
			Index = ObjectIndex;

			ObjectFound = true;
		}
	}
}

void COpenGLRenderer::OnKeyDown(UINT Key)
{
	switch(Key)
	{
		case VK_F1:
			IntersectionPlaneNormal = vec3(1.0f, 0.0f, 0.0f);
			IntersectionPlaneD = 0.0f;
			break;

		case VK_F2:
			IntersectionPlaneNormal = vec3(0.0f, 1.0f, 0.0f);
			IntersectionPlaneD = 0.0f;
			break;

		case VK_F3:
			IntersectionPlaneNormal = vec3(0.0f, 0.0f, 1.0f);
			IntersectionPlaneD = 0.0f;
			break;

		case VK_F4:
			ShowAxisGrid = !ShowAxisGrid;
			break;

		case VK_F5:
			if(EditMode == 3) CollisionDetector.Destroy();
			EditMode = 1;
			break;

		case VK_F6:
			if(EditMode == 3) CollisionDetector.Destroy();
			EditMode = 2;
			break;

		case VK_F7:
			EditMode = 3;
			Camera.Look(Camera.Position, Camera.Reference);
			InitCollisionDetector();
			break;

		case '1':
			IntersectionPointRoundFactor = 1.0f;
			break;

		case '2':
			IntersectionPointRoundFactor = 0.5f;
			break;

		case '3':
			IntersectionPointRoundFactor = 0.25f;
			break;

		case '4':
			IntersectionPointRoundFactor = 0.125f;
			break;

		case '5':
			RenderVertices = !RenderVertices;
			break;

		case '6':
			RenderEdges = !RenderEdges;
			break;

		case '7':
			RenderNormals = !RenderNormals;
			break;

		case 'Q':
			if(IntersectionPlaneNormal.x == 1.0f) Object.rx += 45.0f;
			if(IntersectionPlaneNormal.y == 1.0f) Object.ry += 45.0f;
			if(IntersectionPlaneNormal.z == 1.0f) Object.rz += 45.0f;
			Object.CalculateModelMatrix();
			break;

		case 'E':
			if(IntersectionPlaneNormal.x == 1.0f) Object.rx -= 45.0f;
			if(IntersectionPlaneNormal.y == 1.0f) Object.ry -= 45.0f;
			if(IntersectionPlaneNormal.z == 1.0f) Object.rz -= 45.0f;
			Object.CalculateModelMatrix();
			break;

		case 'T':
			IntersectionPlaneD -= IntersectionPointRoundFactor;
			break;

		case 'G':
			IntersectionPlaneD += IntersectionPointRoundFactor;
			break;

		case 'C':
			if(EditMode != 3) Camera.Move(Scene.GetMiddle() - Camera.Reference);
			if(EditMode == 3) CollisionDetector.Crouch();
			break;

		case 'N':
			if(EditMode != 3)
			{
				Scene.Destroy();
				SceneId = 1;
				while(SceneFileExists(SceneId)) SceneId++;
				SceneFileName.Set("scene%d.scn", SceneId);
				Text = SceneFileName + ", " + ObjectFileName;
			}
			break;

		case 'L':
			if(EditMode != 3) Scene.Load(SceneFileName);
			break;

		case 'V':
			if(EditMode != 3) Scene.Save(SceneFileName);
			break;

		case 'B':
			if(EditMode != 3)
			{
				SceneFileName.Set("scene%d.bin", SceneId);
				Scene.SaveBinary(SceneFileName);
				SceneFileName.Set("scene%d.scn", SceneId);
			}
			break;

		case VK_SPACE:
			if(EditMode == 3)
			{
				CollisionDetector.Jump();
			}
			break;

		case VK_HOME:
			if(EditMode != 3 && SceneFileExists(SceneId - 1))
			{
				Scene.Destroy();
				SceneId--;
				SceneFileName.Set("scene%d.scn", SceneId);
				Scene.Load(SceneFileName);
				Text = SceneFileName + ", " + ObjectFileName;
			}
			break;

		case VK_END:
			if(EditMode != 3 && SceneFileExists(SceneId + 1))
			{
				Scene.Destroy();
				SceneId++;
				SceneFileName.Set("scene%d.scn", SceneId);
				Scene.Load(SceneFileName);
				Text = SceneFileName + ", " + ObjectFileName;
			}
			break;

		case VK_PRIOR:
			if(EditMode != 3 && ObjectFileExists(ObjectId - 1))
			{
				Object.Destroy();
				ObjectId--;
				ObjectFileName.Set("object%d.objs", ObjectId);
				Object.Load(ObjectFileName);
				Text = SceneFileName + ", " + ObjectFileName;
			}
			break;

		case VK_NEXT:
			if(EditMode != 3 && ObjectFileExists(ObjectId + 1))
			{
				Object.Destroy();
				ObjectId++;
				ObjectFileName.Set("object%d.objs", ObjectId);
				Object.Load(ObjectFileName);
				Text = SceneFileName + ", " + ObjectFileName;
			}
			break;

		case VK_NUMPAD6:
			Object.sx += IntersectionPointRoundFactor;
			Object.CalculateModelMatrix();
			break;

		case VK_NUMPAD4:
			Object.sx -= IntersectionPointRoundFactor;
			Object.CalculateModelMatrix();
			break;

		case VK_NUMPAD8:
			Object.sy += IntersectionPointRoundFactor;
			Object.CalculateModelMatrix();
			break;

		case VK_NUMPAD2:
			Object.sy -= IntersectionPointRoundFactor;
			Object.CalculateModelMatrix();
			break;

		case VK_NUMPAD9:
			Object.sz += IntersectionPointRoundFactor;
			Object.CalculateModelMatrix();
			break;

		case VK_NUMPAD1:
			Object.sz -= IntersectionPointRoundFactor;
			Object.CalculateModelMatrix();
			break;

		case VK_NUMPAD5:
			Object.sx = 1.0f;
			Object.sy = 1.0f;
			Object.sz = 1.0f;
			Object.CalculateModelMatrix();
			break;
	}

	FindPrimitive();
}

void COpenGLRenderer::OnLButtonDown(int X, int Y)
{
	LastClickedX = X;
	LastClickedY = Y;
}

void COpenGLRenderer::OnLButtonUp(int X, int Y)
{
	if(X == LastClickedX && Y == LastClickedY)
	{
		if(EditMode == 1 && IntersectionPointFound)
		{
			Scene.AddObject(Object);
		}
	}
}

void COpenGLRenderer::OnMouseMove(int X, int Y)
{
	if(GetKeyState(VK_RBUTTON) & 0x80)
	{
		Camera.OnMouseMove(LastX - X, LastY - Y);
	}

	LastX = X;
	LastY = Y;

	if(EditMode != 3)
	{
		FindPrimitive();
	}
}

void COpenGLRenderer::OnMouseWheel(short zDelta)
{
	if(EditMode != 3)
	{
		Camera.OnMouseWheel(zDelta);

		FindPrimitive();
	}
}

void COpenGLRenderer::OnRButtonDown(int X, int Y)
{
	LastClickedX = X;
	LastClickedY = Y;
}

void COpenGLRenderer::OnRButtonUp(int X, int Y)
{
	if(X == LastClickedX && Y == LastClickedY)
	{
		if(EditMode == 2 && ObjectFound)
		{
			Scene.DeleteObject(Index);

			FindPrimitive();
		}
	}
}

// ----------------------------------------------------------------------------------------------------------------------------

CString ModuleDirectory, ErrorLog;

// ----------------------------------------------------------------------------------------------------------------------------

void GetModuleDirectory()
{
	char *moduledirectory = new char[256];
	GetModuleFileName(GetModuleHandle(NULL), moduledirectory, 256);
	*(strrchr(moduledirectory, '\\') + 1) = 0;
	ModuleDirectory = moduledirectory;
	delete [] moduledirectory;
}

// ----------------------------------------------------------------------------------------------------------------------------

COpenGLView::COpenGLView()
{
}

COpenGLView::~COpenGLView()
{
}

bool COpenGLView::Init(HINSTANCE hInstance, char *Title, int Width, int Height, int Samples)
{
	this->Title = Title;
	this->Width = Width;
	this->Height = Height;

	WNDCLASSEX WndClassEx;

	memset(&WndClassEx, 0, sizeof(WNDCLASSEX));

	WndClassEx.cbSize = sizeof(WNDCLASSEX);
	WndClassEx.style = CS_OWNDC | CS_HREDRAW | CS_VREDRAW;
	WndClassEx.lpfnWndProc = WndProc;
	WndClassEx.hInstance = hInstance;
	WndClassEx.hIcon = LoadIcon(NULL, IDI_APPLICATION);
	WndClassEx.hIconSm = LoadIcon(NULL, IDI_APPLICATION);
	WndClassEx.hCursor = LoadCursor(NULL, IDC_ARROW);
	WndClassEx.lpszClassName = "Win32OpenGLWindowClass";

	if(RegisterClassEx(&WndClassEx) == 0)
	{
		ErrorLog.Set("RegisterClassEx failed!");
		return false;
	}

	DWORD Style = WS_OVERLAPPEDWINDOW | WS_CLIPSIBLINGS | WS_CLIPCHILDREN;

	hWnd = CreateWindowEx(WS_EX_APPWINDOW, WndClassEx.lpszClassName, Title, Style, 0, 0, Width, Height, NULL, NULL, hInstance, NULL);

	if(hWnd == NULL)
	{
		ErrorLog.Set("CreateWindowEx failed!");
		return false;
	}

	HDC hDC = GetDC(hWnd);

	if(hDC == NULL)
	{
		ErrorLog.Set("GetDC failed!");
		return false;
	}

	PIXELFORMATDESCRIPTOR pfd;

	memset(&pfd, 0, sizeof(PIXELFORMATDESCRIPTOR));

	pfd.nSize = sizeof(PIXELFORMATDESCRIPTOR);
	pfd.nVersion = 1;
	pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
	pfd.iPixelType = PFD_TYPE_RGBA;
	pfd.cColorBits = 32;
	pfd.cDepthBits = 24;
	pfd.iLayerType = PFD_MAIN_PLANE;

	int PixelFormat = ChoosePixelFormat(hDC, &pfd);

	if(PixelFormat == 0)
	{
		ErrorLog.Set("ChoosePixelFormat failed!");
		return false;
	}

	static int MSAAPixelFormat = 0;

	if(SetPixelFormat(hDC, MSAAPixelFormat == 0 ? PixelFormat : MSAAPixelFormat, &pfd) == FALSE)
	{
		ErrorLog.Set("SetPixelFormat failed!");
		return false;
	}

	hGLRC = wglCreateContext(hDC);

	if(hGLRC == NULL)
	{
		ErrorLog.Set("wglCreateContext failed!");
		return false;
	}

	if(wglMakeCurrent(hDC, hGLRC) == FALSE)
	{
		ErrorLog.Set("wglMakeCurrent failed!");
		return false;
	}

	if(glewInit() != GLEW_OK)
	{
		ErrorLog.Set("glewInit failed!");
		return false;
	}

	if(!GLEW_VERSION_2_1)
	{
		ErrorLog.Set("OpenGL 2.1 not supported!");
		return false;
	}

	if(MSAAPixelFormat == 0 && Samples > 0)
	{
		if(GLEW_ARB_multisample && WGLEW_ARB_pixel_format)
		{
			while(Samples > 0)
			{
				UINT NumFormats = 0;

				int PFAttribs[] =
				{
					WGL_DRAW_TO_WINDOW_ARB, GL_TRUE,
					WGL_SUPPORT_OPENGL_ARB, GL_TRUE,
					WGL_DOUBLE_BUFFER_ARB, GL_TRUE,
					WGL_PIXEL_TYPE_ARB, WGL_TYPE_RGBA_ARB,
					WGL_COLOR_BITS_ARB, 32,
					WGL_DEPTH_BITS_ARB, 24,
					WGL_ACCELERATION_ARB, WGL_FULL_ACCELERATION_ARB,
					WGL_SAMPLE_BUFFERS_ARB, GL_TRUE,
					WGL_SAMPLES_ARB, Samples,
					0
				};

				if(wglChoosePixelFormatARB(hDC, PFAttribs, NULL, 1, &MSAAPixelFormat, &NumFormats) == TRUE && NumFormats > 0) break;

				Samples--;
			}

			wglDeleteContext(hGLRC);
			DestroyWindow(hWnd);
			UnregisterClass(WndClassEx.lpszClassName, hInstance);

			return Init(hInstance, Title, Width, Height, Samples);
		}
		else
		{
			Samples = 0;
		}
	}

	this->Samples = Samples;

	GetModuleDirectory();

	glGetIntegerv(GL_MAX_TEXTURE_SIZE, &gl_max_texture_size);

	if(GLEW_EXT_texture_filter_anisotropic)
	{
		glGetIntegerv(GL_MAX_TEXTURE_MAX_ANISOTROPY_EXT, &gl_max_texture_max_anisotropy_ext);
	}

	if(WGLEW_EXT_swap_control)
	{
		wglSwapIntervalEXT(0);
	}

	return OpenGLRenderer.Init();
}

void COpenGLView::Show(bool Maximized)
{
	RECT dRect, wRect, cRect;

	GetWindowRect(GetDesktopWindow(), &dRect);
	GetWindowRect(hWnd, &wRect);
	GetClientRect(hWnd, &cRect);

	wRect.right += Width - cRect.right;
	wRect.bottom += Height - cRect.bottom;
	wRect.right -= wRect.left;
	wRect.bottom -= wRect.top;
	wRect.left = dRect.right / 2 - wRect.right / 2;
	wRect.top = dRect.bottom / 2 - wRect.bottom / 2;

	MoveWindow(hWnd, wRect.left, wRect.top, wRect.right, wRect.bottom, FALSE);

	ShowWindow(hWnd, Maximized ? SW_SHOWMAXIMIZED : SW_SHOWNORMAL);
}

void COpenGLView::MessageLoop()
{
	MSG Msg;

	while(GetMessage(&Msg, NULL, 0, 0) > 0)
	{
		TranslateMessage(&Msg);
		DispatchMessage(&Msg);
	}
}

void COpenGLView::Destroy()
{
	if(GLEW_VERSION_2_1)
	{
		OpenGLRenderer.Destroy();
	}

	wglDeleteContext(hGLRC);
	DestroyWindow(hWnd);
}

void COpenGLView::OnKeyDown(UINT Key)
{
	OpenGLRenderer.OnKeyDown(Key);
}

void COpenGLView::OnLButtonDown(int X, int Y)
{
	OpenGLRenderer.OnLButtonDown(X, Y);
}

void COpenGLView::OnLButtonUp(int X, int Y)
{
	OpenGLRenderer.OnLButtonUp(X, Y);
}

void COpenGLView::OnMouseMove(int X, int Y)
{
	OpenGLRenderer.OnMouseMove(X, Y);
}

void COpenGLView::OnMouseWheel(short zDelta)
{
	OpenGLRenderer.OnMouseWheel(zDelta);
}

void COpenGLView::OnPaint()
{
	static DWORD LastFPSTime = GetTickCount(), LastFrameTime = LastFPSTime, FPS = 0;

	PAINTSTRUCT ps;

	HDC hDC = BeginPaint(hWnd, &ps);

	DWORD Time = GetTickCount();

	float FrameTime = (Time - LastFrameTime) * 0.001f;

	LastFrameTime = Time;

	if(Time - LastFPSTime > 1000)
	{
		CString Text = Title;

		if(OpenGLRenderer.Text[0] != 0)
		{
			Text.Append(" - " + OpenGLRenderer.Text);
		}

		Text.Append(" - %dx%d", Width, Height);
		Text.Append(", ATF %dx", gl_max_texture_max_anisotropy_ext);
		Text.Append(", MSAA %dx", Samples);
		Text.Append(", FPS: %d", FPS);
		Text.Append(" - %s", glGetString(GL_RENDERER));

		SetWindowText(hWnd, Text);

		LastFPSTime = Time;
		FPS = 0;
	}
	else
	{
		FPS++;
	}

	OpenGLRenderer.CheckCameraKeys(FrameTime);

	OpenGLRenderer.Render(FrameTime);

	SwapBuffers(hDC);

	EndPaint(hWnd, &ps);

	InvalidateRect(hWnd, NULL, FALSE);
}

void COpenGLView::OnRButtonDown(int X, int Y)
{
	OpenGLRenderer.OnRButtonDown(X, Y);
}

void COpenGLView::OnRButtonUp(int X, int Y)
{
	OpenGLRenderer.OnRButtonUp(X, Y);
}

void COpenGLView::OnSize(int Width, int Height)
{
	this->Width = Width;
	this->Height = Height;

	OpenGLRenderer.Resize(Width, Height);
}

// ----------------------------------------------------------------------------------------------------------------------------

COpenGLView OpenGLView;

// ----------------------------------------------------------------------------------------------------------------------------

LRESULT CALLBACK WndProc(HWND hWnd, UINT uiMsg, WPARAM wParam, LPARAM lParam)
{
	switch(uiMsg)
	{
		case WM_CLOSE:
			PostQuitMessage(0);
			break;

		case WM_KEYDOWN:
			OpenGLView.OnKeyDown((UINT)wParam);
			break;

		case WM_LBUTTONDOWN:
			OpenGLView.OnLButtonDown(LOWORD(lParam), HIWORD(lParam));
			break;

		case WM_LBUTTONUP:
			OpenGLView.OnLButtonUp(LOWORD(lParam), HIWORD(lParam));
			break;

		case WM_MOUSEMOVE:
			OpenGLView.OnMouseMove(LOWORD(lParam), HIWORD(lParam));
			break;

		case 0x020A: // WM_MOUSWHEEL
			OpenGLView.OnMouseWheel(HIWORD(wParam));
			break;

		case WM_PAINT:
			OpenGLView.OnPaint();
			break;

		case WM_RBUTTONDOWN:
			OpenGLView.OnRButtonDown(LOWORD(lParam), HIWORD(lParam));
			break;

		case WM_RBUTTONUP:
			OpenGLView.OnRButtonUp(LOWORD(lParam), HIWORD(lParam));
			break;

		case WM_SIZE:
			OpenGLView.OnSize(LOWORD(lParam), HIWORD(lParam));
			break;

		default:
			return DefWindowProc(hWnd, uiMsg, wParam, lParam);
	}

	return 0;
}

// ----------------------------------------------------------------------------------------------------------------------------

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR sCmdLine, int iShow)
{
	char *AppName = "Scene editor";

	if(OpenGLView.Init(hInstance, AppName, 800, 600, 4))
	{
		OpenGLView.Show();
		OpenGLView.MessageLoop();
	}
	else
	{
		MessageBox(NULL, ErrorLog, AppName, MB_OK | MB_ICONERROR);
	}

	OpenGLView.Destroy();

	return 0;
}
