// ----------------------------------------------------------------------------------------------------------------------------
//
// Version 2.03
//
// ----------------------------------------------------------------------------------------------------------------------------

#include <windows.h>

#ifndef WM_MOUSWHEEL
	#define WM_MOUSWHEEL 0x020A
#endif

#include "glmath.h"
#include "string.h"

#include <gl/glew.h> // http://glew.sourceforge.net/
#include <gl/wglew.h>

#include <FreeImage.h> // http://freeimage.sourceforge.net/

// ----------------------------------------------------------------------------------------------------------------------------

#pragma comment(lib, "opengl32.lib")
#pragma comment(lib, "glu32.lib")
#pragma comment(lib, "glew32.lib")
#pragma comment(lib, "FreeImage.lib")

// ----------------------------------------------------------------------------------------------------------------------------

extern CString ModuleDirectory, ErrorLog;

// ----------------------------------------------------------------------------------------------------------------------------

#define BUFFER_SIZE_INCREMENT 1048576

// ----------------------------------------------------------------------------------------------------------------------------

class CBuffer
{
private:
	BYTE *Buffer;
	int BufferSize;

private:
	int Position;

public:
	CBuffer();
	~CBuffer();

private:
	void SetDefaults();

public:
	void AddData(void *Data, int DataSize);
	void Empty();
	void *GetData();
	int GetDataSize();
};

// ----------------------------------------------------------------------------------------------------------------------------

extern int gl_max_texture_size, gl_max_texture_max_anisotropy_ext;

// ----------------------------------------------------------------------------------------------------------------------------

class CTexture
{
private:
	GLuint Texture;

public:
	CTexture();
	~CTexture();

private:
	void SetDefaults();

public:
	operator GLuint ();

private:
	FIBITMAP *GetBitmap(char *FileName, int &Width, int &Height, int &BPP);

public:
	bool LoadTexture2D(char *FileName);
	bool LoadTextureCubeMap(char **FileNames);
	void Destroy();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CShaderProgram
{
private:
	GLuint VertexShader, FragmentShader;

private:
	GLuint Program;

public:
	GLuint *UniformLocations, *AttribLocations;

public:
	CShaderProgram();
	~CShaderProgram();

private:
	void SetDefaults();

public:
	operator GLuint ();

private:
	GLuint LoadShader(char *FileName, GLenum Type);

public:
	bool Load(char *VertexShaderFileName, char *FragmentShaderFileName);
	void Destroy();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CCamera
{
public:
	vec3 X, Y, Z, Position, Reference;

public:
	mat4x4 ViewMatrix, ViewMatrixInverse, ProjectionMatrix, ProjectionMatrixInverse, ViewProjectionMatrix, ViewProjectionMatrixInverse;

public:
	CCamera();
	~CCamera();

public:
	void Look(const vec3 &Position, const vec3 &Reference, bool RotateAroundReference = false);
	void Move(const vec3 &Movement);
	vec3 OnKeys(BYTE Keys, float FrameTime);
	void OnMouseMove(int dx, int dy);
	void OnMouseWheel(float zDelta);
	void SetPerspective(float fovy, float aspect, float n, float f);

private:
	void CalculateViewMatrix();
};

// ----------------------------------------------------------------------------------------------------------------------------

class CTriangle
{
public:
	vec3 A, B, C, M;
	vec3 AB, BC, CA;
	float LAB, LBC, LCA;
	vec3 N, NH, N1, N2, N3;
	float D, NdotNH, D1, D2, D3;
	vec3 HPNAB, HPNBC, HPNCA;
	float HPDAB, HPDBC, HPDCA;
	vec3 VPNAB, VPNBC, VPNCA;
	float VPDAB, VPDBC, VPDCA;

public:
	CTriangle();
	CTriangle(const vec3 &A, const vec3 &B, const vec3 &C);
	~CTriangle();

public:
	void Set(const vec3 &A, const vec3 &B, const vec3 &C);

public:
	bool Inside(const vec3 &Point);
	bool RayTriangleIntersectionTest(const vec3 &RayOrigin, const vec3 &RayDirection, float &MinDistance, vec3 &IntersectionPoint);
	bool GetHeightAbove(const vec3 &EyePosition, float &MinDistance, float &Height);
	bool GetHeightUnder(const vec3 &EyePosition, float EyeKneeDistance, float &MinDistance, float &Height);
	bool IntersectionTest(const vec3 &EyePositionA, const vec3 &EyePositionB, const vec3 &Direction, float EyeKneeDistance, float ClosestDistance, const vec3 &PN, float PD, float &MinDistance, vec3 &Compensation);
	bool DistanceTest(const vec3 &EyePositionB, float EyeKneeDistance, float ClosestDistance, float &MinDistance, vec3 &Compensation);
};

// ----------------------------------------------------------------------------------------------------------------------------

class CCollisionDetector
{
private:
	CTriangle *Triangles;
	int TrianglesCount;

private:
	float EyeHeight, EyeKneeDistance, ClosestDistance;

private:
	float EH, EHD2, EKD, EKDD2;

private:
	float FallSpeed;
	int CrouchState;

public:
	CCollisionDetector();
	~CCollisionDetector();

private:
	void SetDefaults();

public:
	void Init(vec3 *Vertices, int VerticesCount, float EyeHeight, float EyeKneeDistance, float ClosestDistance);
	void Destroy();

public:
	void Jump();
	void Crouch();

private:
	bool GetHeightAbove(const vec3 &EyePosition, float &MinDistance, float &Height);
	bool GetHeightUnder(const vec3 &EyePosition, float EyeKneeDistance, float &MinDistance, float &Height);
	bool IntersectionTest(const vec3 &EyePositionA, const vec3 &EyePositionB, const vec3 &Direction, float EyeKneeDistance, float ClosestDistance, const vec3 &PN, float PD, float &MinDistance, vec3 &Compensation);
	bool DistanceTest(const vec3 &EyePositionB, float EyeKneeDistance, float ClosestDistance, float &MinDistance, vec3 &Compensation);

public:
	void CheckHorizontalCollision(const vec3 &EyePosition, vec3 &Movement);
	void CheckVerticalCollision(const vec3 &EyePosition, float FrameTime, vec3 &Movement);
};

// ----------------------------------------------------------------------------------------------------------------------------

class CScene
{
private:
	vec3 *Vertices;
	int VerticesCount;

private:
	GLuint VertexBufferObject;

private:
	CTexture Texture;

public:
	CScene();
	~CScene();

private:
	void SetDefaults();

public:
	bool LoadBinary(const char *FileName);
	void Render();
	void Destroy();

public:
	vec3 *GetVertices();
	int GetVerticesCount();
};

// ----------------------------------------------------------------------------------------------------------------------------

class COpenGLRenderer
{
private:
	int LastX, LastY, LastClickedX, LastClickedY;

private:
	int Width, Height;

private:
	CCamera Camera;

private:
	CCollisionDetector CollisionDetector;

private:
	CShaderProgram Preprocess, SSAO, SSAOFilterH, SSAOFilterV, DeferredLighting, FXAA;
	GLuint RotationTexture, ColorBuffer, NormalBuffer, DepthBuffer, SSAOBuffer, SSAOFilterBuffer, FXAABuffer, FBO;

private:
	CScene Scene;

private:
	bool Texturing, Lighting, ApplySSAO, ApplyFXAA;

public:
	CString Text;

public:
	COpenGLRenderer();
	~COpenGLRenderer();

public:
	bool Init();
	void Render();
	void Animate(float FrameTime);
	void Resize(int Width, int Height);
	void Destroy();

public:
	void CheckCameraKeys(float FrameTime);

public:
	void OnKeyDown(UINT Key);
	void OnLButtonDown(int X, int Y);
	void OnLButtonUp(int X, int Y);
	void OnMouseMove(int X, int Y);
	void OnMouseWheel(short zDelta);
	void OnRButtonDown(int X, int Y);
	void OnRButtonUp(int X, int Y);
};

// ----------------------------------------------------------------------------------------------------------------------------

class COpenGLView
{
private:
	char *Title;
	int Width, Height, Samples;
	HWND hWnd;
	HGLRC hGLRC;

private:
	COpenGLRenderer OpenGLRenderer;

public:
	COpenGLView();
	~COpenGLView();

public:
	bool Init(HINSTANCE hInstance, char *Title, int Width, int Height, int Samples);
	void Show(bool Maximized = false);
	void MessageLoop();
	void Destroy();

public:
	void OnKeyDown(UINT Key);
	void OnLButtonDown(int X, int Y);
	void OnLButtonUp(int X, int Y);
	void OnMouseMove(int X, int Y);
	void OnMouseWheel(short zDelta);
	void OnPaint();
	void OnRButtonDown(int X, int Y);
	void OnRButtonUp(int X, int Y);
	void OnSize(int Width, int Height);
};

// ----------------------------------------------------------------------------------------------------------------------------

LRESULT CALLBACK WndProc(HWND hWnd, UINT uiMsg, WPARAM wParam, LPARAM lParam);

// ----------------------------------------------------------------------------------------------------------------------------

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR sCmdLine, int iShow);
