#include "myopenglrenderer.h"

// ----------------------------------------------------------------------------------------------------------------------------

bool In(int Element, int *Array, int ArrayLength)
{
	for(int i = 0; i < ArrayLength; i++)
	{
		if(Array[i] == Element)
		{
			return true;
		}
	}

	return false;
}

// ----------------------------------------------------------------------------------------------------------------------------

CMyOpenGLRenderer::CMyOpenGLRenderer()
{
	Camera = new CFlyingCamera();

	Camera->SetViewMatrixPointer(&ViewMatrix, &ViewMatrixInverse);

	NewIndicesCount = 0;
}

CMyOpenGLRenderer::~CMyOpenGLRenderer()
{
	delete Camera;
}

bool CMyOpenGLRenderer::Init()
{
	bool Error = false;

	Error |= !Font.Init("font-tahoma.bmp");

	if(Error)
	{
		return false;
	}

	Camera->Look(vec3(2.0f), vec3(0.0f));

	glDepthFunc(GL_LEQUAL);

	return true;
}

void CMyOpenGLRenderer::Resize(int Width, int Height)
{
	this->Width = Width;
	this->Height = Height;

	glViewport(0, 0, Width, Height);

	ProjectionMatrix = perspective(45.0f, (float)Width / (float)Height, 0.125f, 512.0f);
	ProjectionMatrixInverse = inverse(ProjectionMatrix);
}

void CMyOpenGLRenderer::Render()
{
	// ------------------------------------------------------------------------------------------------------------------------

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glMatrixMode(GL_PROJECTION);
	glLoadMatrixf(&ProjectionMatrix);

	glMatrixMode(GL_MODELVIEW);
	glLoadMatrixf(&ViewMatrix);

	glEnable(GL_DEPTH_TEST);
	glEnable(GL_CULL_FACE);

	// ------------------------------------------------------------------------------------------------------------------------

	vec3 *Vertices = (vec3*)this->Vertices.GetData();
	int VerticesCount = this->Vertices.GetDataSize() / 12;

	GLuint *Indices = (GLuint*)this->Indices.GetData();
	int IndicesCount = this->Indices.GetDataSize() / 4;

	glEnable(GL_VERTEX_ARRAY);
	glVertexPointer(3, GL_FLOAT, 12, Vertices);

	glColor3f(1.0f, 1.0f, 1.0f);
	glDrawElements(GL_TRIANGLES, IndicesCount, GL_UNSIGNED_INT, Indices);

	glEnable(GL_POLYGON_OFFSET_LINE);
	glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
	glPolygonOffset(-0.5f, -0.5f);
	glColor3f(0.0f, 0.0f, 0.0f);
	glDrawElements(GL_TRIANGLES, IndicesCount, GL_UNSIGNED_INT, Indices);
	glPolygonOffset(0.0f, 0.0f);
	glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
	glDisable(GL_POLYGON_OFFSET_LINE);

	glPointSize(3.0f);
	glColor3f(0.0f, 0.5f, 1.0f);
	glDrawArrays(GL_POINTS, 0, VerticesCount);
	glPointSize(1.0f);

	glDisable(GL_VERTEX_ARRAY);

	// ------------------------------------------------------------------------------------------------------------------------

	glDisable(GL_CULL_FACE);

	// ------------------------------------------------------------------------------------------------------------------------

	Grid.Render();

	// ------------------------------------------------------------------------------------------------------------------------

	glDisable(GL_DEPTH_TEST);

	// ------------------------------------------------------------------------------------------------------------------------

	glMatrixMode(GL_PROJECTION);
	glLoadMatrixf(&ortho(0.0f, (float)Width, (float)Height, 0.0f, 0.0f, 1.0f));

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	CString Text = this->Text;

	Text.Append("Grid.Center = [%.02f, %.02f, %.02f]\n", Grid.Center.x, Grid.Center.y, Grid.Center.z);
	Text.Append("VerticesCount = %d\n", VerticesCount);
	Text.Append("TrianglesCount = %d\n", IndicesCount / 3);
	Text.Append("Left mouse button: Add or select %d more %s to add a triangle\n", 3 - NewIndicesCount, 3 - NewIndicesCount > 1 ? "vetices" : "vertex");
	Text.Append("Right mouse button: Options\n");

	Font.Printf(Text);

	if(Grid.PointFound)
	{
		Text.Set("[%.02f, %.02f, %.02f]", Grid.Point.x, Grid.Point.y, Grid.Point.z);

		Font.Printf(Text, (int)Grid.PointScreenPosition.x + 5, (int)Grid.PointScreenPosition.y - 11);
	}

	// ------------------------------------------------------------------------------------------------------------------------
}

bool CMyOpenGLRenderer::Animate(float FrameTime)
{
	return false;
}

void CMyOpenGLRenderer::Destroy()
{
	Font.Destroy();
	Vertices.Empty();
	Indices.Empty();
}

int CMyOpenGLRenderer::SelectVertex(const vec3 &Point)
{
	vec3 *Vertex = (vec3*)Vertices.GetData();
	int VerticesCount = Vertices.GetDataSize() / 12;

	for(int i = 0; i < VerticesCount; i++)
	{
		if(Point.x >= Vertex->x - 0.001f && Point.x <= Vertex->x + 0.001f)
		{
			if(Point.y >= Vertex->y - 0.001f && Point.y <= Vertex->y + 0.001f)
			{
				if(Point.z >= Vertex->z - 0.001f && Point.z <= Vertex->z + 0.001f)
				{
					return i;
				}
			}
		}

		Vertex++;
	}

	return -1;
}

void CMyOpenGLRenderer::OnCommand(int ID)
{
	switch(ID)
	{
		case ID_GRID_PLANE_YZ:
			Grid.Normal = vec3(1.0f, 0.0f, 0.0f);
			Grid.D = -dot(Grid.Normal, Grid.Center);
			break;

		case ID_GRID_PLANE_XZ:
			Grid.Normal = vec3(0.0f, 1.0f, 0.0f);
			Grid.D = -dot(Grid.Normal, Grid.Center);
			break;

		case ID_GRID_PLANE_XY:
			Grid.Normal = vec3(0.0f, 0.0f, 1.0f);
			Grid.D = -dot(Grid.Normal, Grid.Center);
			break;

		case ID_GRID_UNIT_100:
			Grid.Unit = 10.0f;
			break;

		case ID_GRID_UNIT_10:
			Grid.Unit = 1.0f;
			break;

		case ID_GRID_UNIT_01:
			Grid.Unit = 0.1f;
			break;

		case ID_GRID_RESET:
			Grid.Center = vec3(0.0f);
			Grid.Normal = vec3(0.0f, 1.0f, 0.0f);
			Grid.D = -dot(Grid.Normal, Grid.Center);
			Grid.Unit = 1.0f;
			break;

		case ID_SET_GRID_CENTER:
			Grid.Center = Grid.Point;
			break;

		case ID_DELETE:
			OnDelete();
			break;
	}
	
}

void CMyOpenGLRenderer::OnDelete()
{
	if(SelectedVertexIndex > -1)
	{
		Vertices.DeleteData(SelectedVertexIndex * 12, 12);

		int *TriangleIndices = (int*)Indices.GetData();

		int Position = 0;

		while(TriangleIndices < (int*)(Indices.GetData() + Indices.GetDataSize()))
		{
			if(In(SelectedVertexIndex, TriangleIndices, 3))
			{
				Indices.DeleteData(Position, 12);
			}
			else
			{
				for(int i = 0; i < 3; i++)
				{
					if(TriangleIndices[i] > SelectedVertexIndex)
					{
						TriangleIndices[i]--;
					}
				}

				TriangleIndices += 3;
				Position += 12;
			}
		}

		if(In(SelectedVertexIndex, NewIndices, NewIndicesCount))
		{
			NewIndicesCount = 0;
		}
		else
		{
			for(int i = 0; i < NewIndicesCount; i++)
			{
				if(NewIndices[i] > SelectedVertexIndex)
				{
					NewIndices[i]--;
				}
			}
		}
	}
}

bool CMyOpenGLRenderer::OnCameraKeys(SHORT Keys, float FrameTime, vec3 &Movement)
{
	bool MoveCamera = COpenGLRenderer::OnCameraKeys(Keys, FrameTime, Movement);

	if(MoveCamera)
	{
		Grid.FindPoint(LastX, LastY, Width, Height, Camera->Position, ViewMatrix, ViewMatrixInverse, ProjectionMatrix, ProjectionMatrixInverse);
	}

	return MoveCamera;
}

void CMyOpenGLRenderer::OnKeyDown(UINT Key)
{
	switch(Key)
	{
		case VK_F1:
			Grid.Normal = vec3(1.0f, 0.0f, 0.0f);
			Grid.D = -dot(Grid.Normal, Grid.Center);
			break;

		case VK_F2:
			Grid.Normal = vec3(0.0f, 1.0f, 0.0f);
			Grid.D = -dot(Grid.Normal, Grid.Center);
			break;

		case VK_F3:
			Grid.Normal = vec3(0.0f, 0.0f, 1.0f);
			Grid.D = -dot(Grid.Normal, Grid.Center);
			break;

		case VK_ADD:
			Grid.Center += Grid.Normal * Grid.Unit * 0.5f;
			Grid.D = -dot(Grid.Normal, Grid.Center);
			break;

		case VK_SUBTRACT:
			Grid.Center -= Grid.Normal * Grid.Unit * 0.5f;
			Grid.D = -dot(Grid.Normal, Grid.Center);
			break;

		case 'C':
			Grid.Center = vec3(0.0f);
			Grid.Normal = vec3(0.0f, 1.0f, 0.0f);
			Grid.D = -dot(Grid.Normal, Grid.Center);
			Grid.Unit = 1.0f;
			break;

		case VK_ESCAPE:
			NewIndicesCount = 0;
			break;
	}

	Grid.FindPoint(LastX, LastY, Width, Height, Camera->Position, ViewMatrix, ViewMatrixInverse, ProjectionMatrix, ProjectionMatrixInverse);

	COpenGLRenderer::OnKeyDown(Key);
}

void CMyOpenGLRenderer::OnLButtonDown(int X, int Y)
{
	if(Grid.PointFound)
	{
		SelectedVertexIndex = SelectVertex(Grid.Point);

		if(SelectedVertexIndex == -1)
		{
			NewIndices[NewIndicesCount++] = Vertices.AddData(&Grid.Point, 12) / 12;
		}
		else
		{
			if(!In(SelectedVertexIndex, NewIndices, NewIndicesCount))
			{
				NewIndices[NewIndicesCount++] = SelectedVertexIndex;
			}
		}

		if(NewIndicesCount == 3)
		{
			Indices.AddData(NewIndices, 12);
			NewIndicesCount = 0;
		}
	}

	COpenGLRenderer::OnLButtonDown(X, Y);
}

void CMyOpenGLRenderer::OnMouseMove(int X, int Y, bool LButtonDown, bool RButtonDown)
{
	Grid.FindPoint(X, Y, Width, Height, Camera->Position, ViewMatrix, ViewMatrixInverse, ProjectionMatrix, ProjectionMatrixInverse);

	COpenGLRenderer::OnMouseMove(X, Y, LButtonDown, RButtonDown);
}

void CMyOpenGLRenderer::OnMouseWheel(short zDelta)
{
	if(zDelta > 0)
	{
		Grid.Center += Grid.Normal * Grid.Unit * 0.5f;
	}

	if(zDelta < 0)
	{
		Grid.Center -= Grid.Normal * Grid.Unit * 0.5f;
	}

	Grid.D = -dot(Grid.Normal, Grid.Center);

	Grid.FindPoint(LastX, LastY, Width, Height, Camera->Position, ViewMatrix, ViewMatrixInverse, ProjectionMatrix, ProjectionMatrixInverse);

	COpenGLRenderer::OnMouseWheel(zDelta);
}

void CMyOpenGLRenderer::OnRButtonDown(int X, int Y)
{
	if(Grid.PointFound)
	{
		SelectedVertexIndex = SelectVertex(Grid.Point);
	}

	COpenGLRenderer::OnRButtonDown(X, Y);
}
